package harddelete

import (
	"context"
	"encoding/json"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/twitch-events/gea/internal/db"
	"code.justin.tv/twitch-events/gea/internal/follows"
	"code.justin.tv/web/users-service/models"
	"github.com/aws/aws-sdk-go/service/sqs"
	net_context "golang.org/x/net/context"
)

const (
	followsLimit = 1000
)

type WorkerConfig struct {
	feedsqs.SQSQueueProcessorConfig
}

type Job = models.SNSHardDeleteEvent

type SNSPayload struct {
	Message string `json:"Message"`
}

// Load sets up WorkerConfig to read configuration values from the given distconf.
func (c *WorkerConfig) Load(dconf *distconf.Distconf) error {
	return c.SQSQueueProcessorConfig.Verify(dconf, "gea.hard_delete_user", time.Second*3, 2, time.Minute*15)
}

type Worker struct {
	feedsqs.SQSQueueProcessor
	Config *WorkerConfig

	DBClient           db.DB
	EventFollowsClient *follows.EventFollows
}

func (w *Worker) Setup() error {
	w.SQSQueueProcessor.ProcessMessage = w.processMessageWithStats
	return w.SQSQueueProcessor.Setup()
}

func (w *Worker) processMessageWithStats(ctx net_context.Context, m *sqs.Message) error {
	defer w.recordDuration("worker.process_message.duration", time.Now())
	w.Stats.IncC("worker.process_message.start", 1, 1)

	err := w.processMessage(m)
	if err != nil {
		w.Stats.IncC("worker.process_message.failed", 1, 1)
		return err
	}

	w.Stats.IncC("worker.process_message.succeeded", 1, 1)
	return nil
}

func (w *Worker) processMessage(m *sqs.Message) error {
	if m.Body == nil {
		return errors.New("received empty body message")
	}

	body := &SNSPayload{}
	if err := json.NewDecoder(strings.NewReader(*m.Body)).Decode(body); err != nil {
		return errors.Wrap(err, "unable to parse SQS message body")
	}

	msg := &Job{}
	if err := json.NewDecoder(strings.NewReader(body.Message)).Decode(msg); err != nil {
		return errors.Wrap(err, "unable to parse SNS message body")
	}

	return w.ProcessHardDeleteUserJob(context.Background(), msg)
}

func (w *Worker) ProcessHardDeleteUserJob(ctx context.Context, hardDeleteUserJob *Job) error {
	userID := hardDeleteUserJob.UserID
	ownedEvents, err := w.DBClient.HardDeleteEventsByOwnerID(ctx, userID)
	if err != nil {
		return err
	}

	// Unfollow everyone from the user's events
	for _, event := range ownedEvents {
		var followerIDs []string
		pagedUserIDs := &follows.PagedUserIDs{Cursor: ""}
		for ok := true; ok; ok = pagedUserIDs.Cursor != "" {
			pagedUserIDs, err = w.EventFollowsClient.GetFollowersByEventID(ctx, event.ID, followsLimit, pagedUserIDs.Cursor)
			if err != nil {
				return err
			}
			followerIDs = append(followerIDs, pagedUserIDs.UserIDs...)
		}

		for _, followerID := range followerIDs {
			_, err = w.EventFollowsClient.Unfollow(ctx, event.ID, followerID)
			if err != nil {
				return err
			}
		}
	}

	// Unfollow the user from the events they follow
	var followedEventIDs []string
	followedEvents := &follows.PagedEventIDs{Cursor: ""}
	for ok := true; ok; ok = followedEvents.Cursor != "" {
		followedEvents, err = w.EventFollowsClient.GetFollowedEventsByUserID(ctx, userID, followsLimit, followedEvents.Cursor)
		if err != nil {
			return err
		}
		followedEventIDs = append(followedEventIDs, followedEvents.EventIDs...)
	}

	for _, followedEventID := range followedEventIDs {
		success, err := w.EventFollowsClient.Unfollow(ctx, followedEventID, userID)
		if err != nil {
			return err
		}
		if success {
			_, err = w.DBClient.DecrementEventFollowCount(ctx, followedEventID)
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func (w *Worker) recordDuration(stat string, start time.Time) {
	w.Stats.TimingDurationC(stat, time.Since(start), 1.0)
}
