package video

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/vod/vodapi/rpc/vodapi"
	vodapi_utils "code.justin.tv/vod/vodapi/rpc/vodapi/utils"
)

// Broadcast-type filters that Gala accepts.
const (
	BroadcastTypeHighlight = "highlight"
	BroadcastTypeArchive   = "archive"
)

type ArchiveVideo struct {
	VodID         string    `json:"vod_id"`
	OffsetSeconds int       `json:"offset_seconds"`
	StartTime     time.Time `json:"start_time"`
	BroadcastType string    `json:"broadcast_type"`
	Views         int       `json:"views"`
}

type FinderConfig struct {
	VodAPIURL             *distconf.Str
	VideoLookbackDuration *distconf.Duration
	DefaultVodAPILimit    *distconf.Int
	MaxVodAPIRecords      *distconf.Int
	MaxArchivesReturned   *distconf.Int
	MaxHighlightsReturned *distconf.Int
}

func (c *FinderConfig) Load(dconf *distconf.Distconf) error {
	c.VodAPIURL = dconf.Str("gea.video.vodapi_url", "")
	if c.VodAPIURL.Get() == "" {
		return errors.New("unable to find vodapi URL")
	}
	c.VideoLookbackDuration = dconf.Duration("gea.video.video_lookback_duration", 2*24*time.Hour)
	c.DefaultVodAPILimit = dconf.Int("gea.video.default_vodapi_limit", 100)
	c.MaxVodAPIRecords = dconf.Int("gea.video.max_vodapi_records", 300)
	c.MaxArchivesReturned = dconf.Int("gea.video.max_archives_returned", 2)
	c.MaxHighlightsReturned = dconf.Int("gea.video.max_highlights_returned", 5)

	return nil
}

type VodAPIClient interface {
	PublicGetVodsByUser(context.Context, *vodapi.PublicGetVodsByUserRequest) (*vodapi.PublicGetVodsByUserResponse, error)
}

type Finder struct {
	Config *FinderConfig
	VodAPI VodAPIClient
}

// Loads a list of archive and highlight videos that ran on ONE user's channel over ONE time period.
//
// The list of videos will not be exhaustive, but should be the current most compelling set of videos that can be
// loaded in a short amount of time.
// This list may be combined with other lists of videos fetched for other channels and time periods to create a
// list of videos that completely cover an event.
func (v *Finder) FindArchiveVideos(ctx context.Context, userID string, startTime time.Time, endTime time.Time) ([]ArchiveVideo, error) {
	vods, err := v.loadArchiveAndHighlightVods(ctx, userID, startTime, endTime)
	if err != nil {
		return nil, err
	}

	archiveVideos := make([]ArchiveVideo, len(vods))
	for i, vod := range vods {
		archiveVideos[i] = vodToArchiveVideo(vod, startTime)
	}

	return archiveVideos, nil
}

// Try to load both archives and highlights that have the most views, that overlap with the event.
func (v *Finder) loadArchiveAndHighlightVods(ctx context.Context, userID string, startTime time.Time, endTime time.Time) ([]*vodapi.Vod, error) {
	maxNumArchives := int(v.Config.MaxArchivesReturned.Get())
	maxNumHighlights := int(v.Config.MaxHighlightsReturned.Get())
	maxVodAPIRecordsToSearch := int(v.Config.MaxVodAPIRecords.Get())

	limit := int(v.Config.DefaultVodAPILimit.Get())
	offset := 0
	numHighlights := 0
	numArchives := 0

	recordedAfter := startTime.Add(-1 * v.Config.VideoLookbackDuration.Get())
	recordedBefore := endTime

	vods := make([]*vodapi.Vod, 0, maxNumArchives+maxNumHighlights)

	for {
		// If we've loaded enough archives and highlights, exit.
		if numArchives >= maxNumArchives && numHighlights >= maxNumHighlights {
			break
		}
		// If the number of VODs we've loaded passes a threshold, end the search.
		if offset >= maxVodAPIRecordsToSearch {
			break
		}

		resp, err := v.VodAPI.PublicGetVodsByUser(ctx, &vodapi.PublicGetVodsByUserRequest{
			ChannelId:      userID,
			BroadcastTypes: []vodapi.VodType{vodapi.VodType_ARCHIVE, vodapi.VodType_HIGHLIGHT},
			Sort:           vodapi.VodSort_VIEWS,
			RecordedAfter:  vodapi_utils.ProtobufTimeAsTimestamp(&recordedAfter),
			RecordedBefore: vodapi_utils.ProtobufTimeAsTimestamp(&recordedBefore),
			Limit:          int64(limit),
			Offset:         int64(offset),
		})
		if err != nil {
			return nil, errors.Wrap(err, "could not find videos")
		}
		// If the search has no results, end the search.
		if len(resp.Vods) == 0 {
			break
		}

		for _, vod := range resp.Vods {
			if numArchives >= maxNumArchives && numHighlights >= maxNumHighlights {
				break
			}

			// Skip VODs that don't overlap with our event.
			if !overlap(startTime, endTime, vod) {
				continue
			}

			if vod.BroadcastType == vodapi.VodType_ARCHIVE && numArchives < maxNumArchives {
				numArchives++
				vods = append(vods, vod)
			} else if vod.BroadcastType == vodapi.VodType_HIGHLIGHT && numHighlights < maxNumHighlights {
				numHighlights++
				vods = append(vods, vod)
			}
		}

		offset += limit
	}

	return vods, nil
}

func overlap(startTime time.Time, endTime time.Time, vod *vodapi.Vod) bool {
	vodStartTime := vodapi_utils.FromProtobufTimestampToTime(vod.StartedOn)
	vodEndTime := vodStartTime.Add(time.Duration(vod.TotalLength) * time.Second)

	return !(vodStartTime.After(endTime) || vodEndTime.Before(startTime))
}

func vodToArchiveVideo(vod *vodapi.Vod, startTime time.Time) ArchiveVideo {
	var vodStartTime time.Time
	if vod.StartedOn != nil {
		vodStartTime = vodapi_utils.FromProtobufTimestampToTime(vod.StartedOn).UTC()
	}
	offsetSeconds := 0
	if vodStartTime.Before(startTime) {
		offsetSeconds = int(startTime.Sub(vodStartTime).Seconds())
		vodStartTime = startTime
	}

	return ArchiveVideo{
		VodID:         vod.Id,
		OffsetSeconds: offsetSeconds,
		StartTime:     vodStartTime,
		BroadcastType: vodapi.ConvertVodType(vod.BroadcastType),
		Views:         int(vod.Views),
	}
}
