package video

import (
	"context"
	"encoding/json"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/twitch-events/gea/internal/cache"
)

type CacheConfig struct {
	CacheDuration *distconf.Duration
}

func (c *CacheConfig) Load(dconf *distconf.Distconf) error {
	c.CacheDuration = dconf.Duration("gea.video.cache_duration", 5*time.Minute)
	return nil
}

type Cache struct {
	Config *CacheConfig
	Cache  *cache.Service
	Log    *log.ElevatedLog
}

func (c *Cache) archiveKey(eventID string) string {
	// AV stands for archive video
	return "gea:av:" + eventID
}

func (c *Cache) GetArchives(ctx context.Context, eventIDs []string) map[string][]ArchiveVideo {
	keys := make([]string, len(eventIDs))
	idMap := make(map[string]string, len(eventIDs))
	for i, eventID := range eventIDs {
		key := c.archiveKey(eventID)
		keys[i] = key
		idMap[key] = eventID
	}
	cached, err := c.Cache.GetMulti(ctx, keys)
	if err != nil {
		c.Log.Log("err", err, "keys", keys, "could not get archives from video cache")
		return nil
	}
	events := make(map[string][]ArchiveVideo, len(eventIDs))
	for key, item := range cached {
		var videos []ArchiveVideo
		err := json.Unmarshal(item.Value, &videos)
		if err != nil {
			c.Log.Log("err", err, "key", key, "could not unmarshall archives from video cache")
			continue
		}
		if videos != nil {
			events[idMap[key]] = videos
		}
	}
	return events
}

func (c *Cache) CacheArchives(ctx context.Context, eventID string, videos []ArchiveVideo) {
	key := c.archiveKey(eventID)
	data, err := json.Marshal(videos)
	if err != nil {
		c.Log.Log("err", err, "key", key, "could not marshall archives for video cache")
		return
	}
	cacheErr := c.Cache.Set(ctx, &memcache.Item{
		Key:        key,
		Value:      data,
		Expiration: int32(c.Config.CacheDuration.Get().Seconds()),
	})
	if cacheErr != nil {
		c.Log.Log("err", cacheErr, "key", key, "could not cache archive videos")
		return
	}
}

func (c *Cache) InvalidateVideos(ctx context.Context, eventID string) {
	key := c.archiveKey(eventID)
	err := c.Cache.Delete(ctx, key)
	if err != nil {
		if err != memcache.ErrCacheMiss {
			c.Log.Log("err", err, "key", key, "could not invalidate videos in cache")
		}
	}
}
