package video

import (
	"sort"
	"strings"

	"code.justin.tv/twitch-events/gea/internal/cursor"
)

const (
	SortByStartTime = "start_time"
	SortByViews     = "views"
)

type GetPageArgs struct {
	AfterCursor           string
	Limit                 int
	SortBy                string
	FilterByBroadcastType string
}

type VideosPage struct {
	Items       []VideosPageItem
	HasNextPage bool
}

type VideosPageItem struct {
	Video  *ArchiveVideo
	Cursor string
}

// GetPageOfVideos takes the full list of videos that cover an event, creates a sub-list by filtering and
// sorting the full list, and returns a page of videos out of the sub-list.
func GetPageOfVideos(videos []ArchiveVideo, args *GetPageArgs) (*VideosPage, error) {
	filteredVideos := make([]ArchiveVideo, 0, len(videos))

	filterByBroadcastType := strings.ToLower(args.FilterByBroadcastType)
	shouldFilter := filterByBroadcastType != ""

	for _, video := range videos {
		if shouldFilter && filterByBroadcastType != strings.ToLower(video.BroadcastType) {
			continue
		}
		filteredVideos = append(filteredVideos, video)
	}

	sort.SliceStable(filteredVideos, func(i, j int) bool {
		if args.SortBy == SortByViews {
			// Sort by views will show highest # of views first
			return filteredVideos[i].Views > filteredVideos[j].Views
		}
		return filteredVideos[i].StartTime.Before(filteredVideos[j].StartTime)
	})

	startAt := 0
	if args.AfterCursor != "" {
		afterIndex, err := unpackVideosCursor(args.AfterCursor)
		if err != nil {
			return nil, err
		}
		startAt = afterIndex + 1
	}

	items := make([]VideosPageItem, 0, args.Limit)
	for offset := 0; offset < args.Limit; offset++ {
		i := startAt + offset
		if i >= len(filteredVideos) {
			break
		}

		cursor, err := encodeVideosCursor(i)
		if err != nil {
			return nil, err
		}

		items = append(items, VideosPageItem{
			Video:  &filteredVideos[i],
			Cursor: cursor,
		})
	}

	hasNextPage := startAt+args.Limit < len(filteredVideos)

	return &VideosPage{
		Items:       items,
		HasNextPage: hasNextPage,
	}, nil
}

// Increment when we change the shape of the video cursor.
const currentVideosCursorVersion = 1

func encodeVideosCursor(index int) (string, error) {
	c := cursor.IndexCursor{
		Version: currentVideosCursorVersion,
		Index:   index,
	}
	return cursor.Encode(&c)
}

func unpackVideosCursor(cursorStr string) (int, error) {
	c := cursor.IndexCursor{}
	_, err := cursor.DecodeVersioned(cursorStr, currentVideosCursorVersion, &c)
	if err != nil {
		return 0, err
	}

	return c.Index, nil
}
