package geaclient

import (
	"context"
	"time"

	"code.justin.tv/foundation/twitchclient"
)

type SegmentEvent struct {
	ID       string `json:"id"`
	OwnerID  string `json:"owner_id"`
	Type     string `json:"type"`
	ParentID string `json:"parent_id"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at"`

	StartTime time.Time `json:"start_time"`
	EndTime   time.Time `json:"end_time"`
	// TimeZoneID is a tz database time zone name.  E.g. America/New_York
	TimeZoneID string `json:"time_zone_id"`

	ImageID     *string `json:"image_id"`
	ImageURL    string  `json:"image_url"`
	Language    string  `json:"language"`
	Title       string  `json:"title"`
	Description string  `json:"description"`
	ChannelID   string  `json:"channel_id"`
	GameID      string  `json:"game_id"`
}

func (e SegmentEvent) GetID() string {
	return e.ID
}

func (e SegmentEvent) GetType() string {
	return e.Type
}

func (e *SegmentEvent) fromSuperstruct(s *eventSuperstruct) error {
	err := validateNotNil(map[string]interface{}{
		"parent_id":    s.ParentID,
		"start_time":   s.StartTime,
		"end_time":     s.EndTime,
		"time_zone_id": s.TimeZoneID,
		"image_url":    s.ImageURL,
		"language":     s.Language,
		"title":        s.Title,
		"description":  s.Description,
		"channel_id":   s.ChannelID,
		"game_id":      s.GameID,
	})
	if err != nil {
		return err
	}
	e.ID = s.ID
	e.OwnerID = s.OwnerID
	e.Type = EventTypeSegment
	e.ParentID = *s.ParentID

	e.CreatedAt = s.CreatedAt
	e.UpdatedAt = s.UpdatedAt
	e.DeletedAt = s.DeletedAt

	e.StartTime = *s.StartTime
	e.EndTime = *s.EndTime
	e.TimeZoneID = *s.TimeZoneID

	e.ImageID = s.ImageID
	e.ImageURL = *s.ImageURL
	e.Language = *s.Language
	e.Title = *s.Title
	e.Description = *s.Description
	e.ChannelID = *s.ChannelID
	e.GameID = *s.GameID

	return nil
}

type CreateSegmentEventParams struct {
	OwnerID  string
	ParentID string

	StartTime time.Time
	EndTime   time.Time

	ImageID     *string
	Title       string
	Description string
	ChannelID   string
	GameID      string
}

func (c *ClientImpl) CreateSegmentEvent(ctx context.Context, params CreateSegmentEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SegmentEvent, error) {
	var response SegmentEvent
	err := c.createEvent(ctx, "create_segment_event", createEventParams{
		OwnerID:     params.OwnerID,
		Type:        EventTypeSegment,
		ParentID:    &params.ParentID,
		StartTime:   &params.StartTime,
		EndTime:     &params.EndTime,
		ImageID:     params.ImageID,
		Title:       &params.Title,
		Description: &params.Description,
		ChannelID:   &params.ChannelID,
		GameID:      &params.GameID,
	}, &response, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type UpdateSegmentEventParams struct {
	OwnerID *string

	StartTime *time.Time
	EndTime   *time.Time

	ImageID     *string
	Title       *string
	Description *string
	ChannelID   *string
	GameID      *string
}

func (c *ClientImpl) UpdateSegmentEvent(ctx context.Context, eventID string, params UpdateSegmentEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SegmentEvent, error) {
	var response SegmentEvent
	err := c.updateEvent(ctx, "update_segment_event", updateEventParams{
		ID:          eventID,
		OwnerID:     params.OwnerID,
		StartTime:   params.StartTime,
		EndTime:     params.EndTime,
		ImageID:     params.ImageID,
		Title:       params.Title,
		Description: params.Description,
		ChannelID:   params.ChannelID,
		GameID:      params.GameID,
	}, &response, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}
