package geaclient

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
)

type SingleEvent struct {
	ID      string `json:"id"`
	OwnerID string `json:"owner_id"`
	Type    string `json:"type"`

	CreatedAt time.Time  `json:"created_at"`
	UpdatedAt time.Time  `json:"updated_at"`
	DeletedAt *time.Time `json:"deleted_at"`

	StartTime time.Time `json:"start_time"`
	EndTime   time.Time `json:"end_time"`
	// TimeZoneID is a tz database time zone name.  E.g. America/New_York
	// TimeZoneID may be an empty string until it is fully supported by Gea.
	TimeZoneID string `json:"time_zone_id"`

	ImageID     *string `json:"image_id"`
	ImageURL    string  `json:"image_url"`
	Language    string  `json:"language"`
	Title       string  `json:"title"`
	Description string  `json:"description"`
	ChannelID   string  `json:"channel_id"`
	GameID      string  `json:"game_id"`
}

func (e SingleEvent) GetID() string {
	return e.ID
}

func (e SingleEvent) GetType() string {
	return e.Type
}

func (e *SingleEvent) fromSuperstruct(s *eventSuperstruct) error {
	err := validateNotNil(map[string]interface{}{
		"start_time":  s.StartTime,
		"end_time":    s.EndTime,
		"language":    s.Language,
		"title":       s.Title,
		"description": s.Description,
		"image_url":   s.ImageURL,
		"channel_id":  s.ChannelID,
		"game_id":     s.GameID,
	})
	if err != nil {
		return err
	}
	e.ID = s.ID
	e.OwnerID = s.OwnerID
	e.Type = EventTypeSingle

	e.CreatedAt = s.CreatedAt
	e.UpdatedAt = s.UpdatedAt
	e.DeletedAt = s.DeletedAt

	e.StartTime = *s.StartTime
	e.EndTime = *s.EndTime

	// TimeZoneID may be nil or an empty string until it is fully supported by Gea.
	if s.TimeZoneID != nil {
		e.TimeZoneID = *s.TimeZoneID
	}

	e.ImageID = s.ImageID
	e.ImageURL = *s.ImageURL
	e.Language = *s.Language
	e.Title = *s.Title
	e.Description = *s.Description
	e.ChannelID = *s.ChannelID
	e.GameID = *s.GameID

	return nil
}

type CreateSingleEventParams struct {
	OwnerID string

	StartTime time.Time
	EndTime   time.Time
	// TimeZoneID is expected to be a tz database time zone name.  E.g. America/New_York
	TimeZoneID string

	ImageID     *string
	Language    string
	Title       string
	Description string
	ChannelID   string
	GameID      string
}

func (c *ClientImpl) CreateSingleEvent(ctx context.Context, params CreateSingleEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SingleEvent, error) {
	var response SingleEvent
	err := c.createEvent(ctx, "create_single_event", createEventParams{
		OwnerID:     params.OwnerID,
		Type:        EventTypeSingle,
		StartTime:   &params.StartTime,
		EndTime:     &params.EndTime,
		TimeZoneID:  &params.TimeZoneID,
		ImageID:     params.ImageID,
		Language:    &params.Language,
		Title:       &params.Title,
		Description: &params.Description,
		ChannelID:   &params.ChannelID,
		GameID:      &params.GameID,
	}, &response, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

type UpdateSingleEventParams struct {
	OwnerID *string

	StartTime *time.Time
	EndTime   *time.Time
	// TimeZoneID is expected to be a tz database time zone name.  E.g. America/New_York
	TimeZoneID *string

	ImageID     *string
	Language    *string
	Title       *string
	Description *string
	ChannelID   *string
	GameID      *string
}

func (c *ClientImpl) UpdateSingleEvent(ctx context.Context, eventID string, params UpdateSingleEventParams, userID string, reqOpts *twitchclient.ReqOpts) (*SingleEvent, error) {
	var response SingleEvent
	err := c.updateEvent(ctx, "update_single_event", updateEventParams{
		ID:          eventID,
		OwnerID:     params.OwnerID,
		StartTime:   params.StartTime,
		EndTime:     params.EndTime,
		TimeZoneID:  params.TimeZoneID,
		ImageID:     params.ImageID,
		Language:    params.Language,
		Title:       params.Title,
		Description: params.Description,
		ChannelID:   params.ChannelID,
		GameID:      params.GameID,
	}, &response, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	return &response, nil
}

func (c *ClientImpl) DeleteSingleEvent(ctx context.Context, eventID string, userID string, reqOpts *twitchclient.ReqOpts) (*SingleEvent, error) {
	event, err := c.DeleteEvent(ctx, eventID, userID, reqOpts)
	if err != nil {
		return nil, err
	}
	if singleEvent, ok := event.(*SingleEvent); ok {
		return singleEvent, nil
	}
	return nil, errors.Errorf("event %s type should be single but was %s", event.GetID(), event.GetType())
}
