terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "tfstate/events/gea/terraform/canary"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

locals {
  name = "gea"
}

# Set the url of the jenkins runner to target, and optionally override the cred file location
provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${local.account}"
    key     = "tfstate/${local.team}/core/terraform/${local.environment}"
    region  = "${local.region}"
    profile = "${local.account}"
  }
}

data "terraform_remote_state" "gea_production" {
  backend = "s3"

  config {
    bucket  = "${local.account}"
    key     = "tfstate/${local.team}/${local.name}/terraform/production"
    region  = "${local.region}"
    profile = "${local.account}"
  }
}

module "gea" {
  source = "../../../modules/gea"

  # Environment
  account                  = "${local.account}"
  environment              = "${local.environment}"
  region                   = "${local.region}"
  ssm_iam_policy_role_name = "${local.ssm_iam_policy_role_name}"

  # Service properties
  desired_count = "2"
  max_size      = "3"
  min_size      = "1"
  task_cpu      = "1024"
  task_mem      = "512"

  # Pipeline properties
  promote_from = "staging"
  auto_promote = "false"

  name                 = "${local.name}"
  owner                = "${local.owner}"
  team                 = "${local.team}"
  alb_target_group_arn = "${data.terraform_remote_state.gea_production.target_group_arn}"

  sandstorm_policy_arn = "${data.terraform_remote_state.core.sandstorm_policy_arn}"
}

module "consulconf_uploads_sqs_queue_url" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.5.1"
  service     = "${local.name}"
  team        = "${local.team}"
  environment = "${local.environment}"
  key         = "uploads.sqssource.queue_url"
  value       = "${data.terraform_remote_state.gea_production.uploads_sqs_queue_url}"
}

module "consulconf_uploads_sns_topic_arn" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.5.1"
  service     = "${local.name}"
  team        = "${local.team}"
  environment = "${local.environment}"
  key         = "uploads.callback_arn"
  value       = "${data.terraform_remote_state.gea_production.uploads_sns_topic_arn}"
}

module "consulconf_uploads_s3_bucket" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.5.1"
  service     = "${local.name}"
  team        = "${local.team}"
  environment = "${local.environment}"
  key         = "uploads.s3bucket"
  value       = "${data.terraform_remote_state.gea_production.uploads_s3_bucket}"
}

module "consulconf_hard_delete_user_queue" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.9.0"
  service     = "${local.name}"
  team        = "${local.team}"
  environment = "${local.environment}"
  key         = "hard_delete_user.sqssource.queue_url"
  value       = "${data.terraform_remote_state.gea_production.hard_delete_user_sqs_queue_url}"
}
