terraform {
  backend "s3" {
    bucket  = "twitch-feed-dev"
    key     = "tfstate/events/gea/terraform/integration"
    region  = "us-west-2"
    profile = "twitch-feed-dev"
  }
}

locals {
  name = "gea"
}

# Set the url of the jenkins runner to target, and optionally override the cred file location
provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${local.account}"
    key     = "tfstate/${local.team}/core/terraform/${local.environment}"
    region  = "${local.region}"
    profile = "${local.account}"
  }
}

module "alb" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=622157fac36fb40cedcdf9e0b40ca6e5fa1b9917"
  service     = "${local.name}"
  owner       = "${local.owner}"
  environment = "${local.environment}"
  account     = "${local.account}"
  team        = "${local.team}"
}

module "gea" {
  source = "../../../modules/gea"

  # Environment
  account                  = "${local.account}"
  environment              = "${local.environment}"
  region                   = "${local.region}"
  ssm_iam_policy_role_name = "${local.ssm_iam_policy_role_name}"

  # Service properties
  desired_count = "2"
  max_size      = "3"
  min_size      = "1"
  task_cpu      = "1024"
  task_mem      = "512"

  # Pipeline properties
  promote_from = "latest"
  auto_promote = "true"

  name                 = "${local.name}"
  owner                = "${local.owner}"
  team                 = "${local.team}"
  alb_target_group_arn = "${module.alb.target_group_arn}"

  sandstorm_policy_arn = "${data.terraform_remote_state.core.sandstorm_policy_arn}"
}




# Scheduled ECS task (and related resources) that periodically sends a request to the create_channel_event_updates
# endpoint.
module "channel-event-poller" {
  source = "../../../modules/poller"

  team        = "${local.team}"
  environment = "${local.environment}"
  name        = "channel-event"

  rate           = "rate(1 minute)"
  gea_endpoint   = "http://events-integration.internal.justin.tv/v1/create_channel_event_updates"
  ecs_cluster_id = "${data.terraform_remote_state.core.common_cluster_id}"
}

module "hard_delete_user" {
  source = "../../../modules/hard_delete_user"

  team                     = "${local.team}"
  environment              = "${local.environment}"
  name                     = "${local.name}"
  hard_delete_user_sns_arn = "arn:aws:sns:us-west-2:465369119046:notification-user-destroyed"
}
