terraform {
  backend "s3" {
    bucket  = "twitch-feed-dev"
    key     = "tfstate/events/gea/terraform/staging"
    region  = "us-west-2"
    profile = "twitch-feed-dev"
  }
}

locals {
  name = "gea"
}

# Set the url of the jenkins runner to target, and optionally override the cred file location
provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/${var.team}/core/terraform/${var.environment}"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/feed/terraform/accounts/${var.account}"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}

module "alb" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=622157fac36fb40cedcdf9e0b40ca6e5fa1b9917"
  service     = "${local.name}"
  owner       = "${var.owner}"
  environment = "${var.environment}"
  account     = "${var.account}"
  team        = "${var.team}"
}

module "http-alb-healthchecks" {
  source   = "../../../modules/http-alb-healthchecks"
  account  = "${var.account}"
  region   = "${var.region}"
  alb_name = "${var.team}-${local.name}-${var.environment}"

  # This is the partial relative-id needed from the ALB's full ARN
  # arn:aws:elasticloadbalancing:us-west-2:724951484461:loadbalancer/app/events-gea-staging/f99b036e2fd330a2
  alb_arn = "app/events-gea-staging/f99b036e2fd330a2"

  # The amount of seconds a condition must be true for a Cloudwatch alarm to fire
  check_period = 60

  # Number of 5xx's that must be recorded in a given time window for the alarm to fire
  rate_5xx_threshold = 10

  # Average latency (ms) above which a Cloudwatch alarm will fire
  latency_threshold = 5000

  # ARN for SNS topic to be notified when a Cloudwatch alarm fires
  action_arn = "arn:aws:sns:us-west-2:724951484461:pagerduty_events"
}

module "gea" {
  source = "../../../modules/gea"

  # Environment
  account                  = "${var.account}"
  environment              = "${var.environment}"
  region                   = "${var.region}"
  ssm_iam_policy_role_name = "${var.ssm_iam_policy_role_name}"

  # Service properties
  desired_count = "2"
  max_size      = "3"
  min_size      = "1"
  task_cpu      = "1024"
  task_mem      = "512"

  # Pipeline properties
  promote_from = "integration"
  auto_promote = "true"

  name                 = "${local.name}"
  owner                = "${var.owner}"
  team                 = "${var.team}"
  alb_target_group_arn = "${module.alb.target_group_arn}"

  sandstorm_policy_arn = "${data.terraform_remote_state.core.sandstorm_policy_arn}"
}

module "lambda" {
  source = "../../../modules/lambda"

  # Environment
  name            = "${local.name}"
  team            = "${var.team}"
  environment     = "${var.environment}"
  rate            = "rate(1 minute)"
  gea_endpoint    = "http://${module.alb.dns_name}/v1/create_notification_jobs"
  subnets         = "${data.terraform_remote_state.account.private_subnets}"
  security_groups = "${data.terraform_remote_state.account.twitch_subnets_sg}"
}

# Scheduled ECS task (and related resources) that periodically sends a request to the create_channel_event_updates
# endpoint.
module "channel-event-poller" {
  source = "../../../modules/poller"

  team        = "${var.team}"
  environment = "${var.environment}"
  name        = "channel-event"

  rate           = "rate(1 minute)"
  gea_endpoint   = "http://events-staging.internal.justin.tv/v1/create_channel_event_updates"
  ecs_cluster_id = "${data.terraform_remote_state.core.common_cluster_id}"
}

module "hard_delete_user" {
  source = "../../../modules/hard_delete_user"

  team                     = "${var.team}"
  environment              = "${var.environment}"
  name                     = "${local.name}"
  hard_delete_user_sns_arn = "arn:aws:sns:us-west-2:465369119046:notification-user-destroyed"
}
