# Module for defining the dynamodb tables

# Environment

variable "account" {}
variable "name" {}
variable "team" {}
variable "environment" {}

# Read and Write Capacity for the Event Follows Table

variable "event_follows_read_capacity" {}
variable "event_follows_write_capacity" {}

# Read and Write Capacity for the notifications table

variable "event_notifications_read_capacity" {}
variable "event_notifications_write_capacity" {}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

# Follows
resource "aws_dynamodb_table" "event_follows_dynamodb" {
  name           = "${var.team}_${var.environment}_event_follows"
  read_capacity  = "${var.event_follows_read_capacity}"
  write_capacity = "${var.event_follows_write_capacity}"
  hash_key       = "event_id"
  range_key      = "user_id"

  attribute {
    name = "event_id"
    type = "S"
  }

  attribute {
    name = "user_id"
    type = "S"
  }

  global_secondary_index {
    name            = "user_id-event_id-index"
    projection_type = "ALL"
    hash_key        = "user_id"
    range_key       = "event_id"
    read_capacity   = "${var.event_follows_read_capacity}"
    write_capacity  = "${var.event_follows_write_capacity}"
  }

  lifecycle {
    prevent_destroy = true
  }

  point_in_time_recovery {
    enabled = true
  }
}

module "event_follows_table_consul_key" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.5.1"
  service     = "${var.name}"
  team        = "${var.team}"
  environment = "${var.environment}"
  key         = "event_follows_table"
  value       = "${aws_dynamodb_table.event_follows_dynamodb.name}"
}

# DynamoDB table to hold whether notifications were sent for events
resource "aws_dynamodb_table" "notifications_dynamodb" {
  name           = "${var.team}_${var.environment}_notifications"
  read_capacity  = "${var.event_notifications_read_capacity}"
  write_capacity = "${var.event_notifications_write_capacity}"
  hash_key       = "event_id"

  attribute {
    name = "event_id"
    type = "S"
  }

  ttl {
    attribute_name = "expires_at"
    enabled        = true
  }

  lifecycle {
    prevent_destroy = true
  }

  point_in_time_recovery {
    enabled = true
  }
}

# Consul key that contains the name of the notifications table, for services to reference
module "notifications_table_consul_conf" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.5.1"
  service     = "${var.name}"
  team        = "${var.team}"
  environment = "${var.environment}"
  key         = "notifications_table"
  value       = "${aws_dynamodb_table.notifications_dynamodb.name}"
}
