# Module for provisioning the AWS resources and Jenkins pipeline for the events/gea service.

# Inputs

## Environment ##
variable "account" {}

variable "name" {}
variable "owner" {}
variable team {}

variable "environment" {}
variable region {}

variable "ssm_iam_policy_role_name" {
  description = "Role name of IAM role to be attached to Gea task"
}

## Service properties
variable "desired_count" {}

variable "max_size" {}
variable "min_size" {}
variable "task_cpu" {}
variable "task_mem" {}

## Pipeline properties
variable auto_promote {}

variable promote_from {}

variable alb_target_group_arn {}

variable "sandstorm_policy_arn" {
  description = "The ARN of the IAM policy that allows a service to assume the Sandstorm role required to load secrets."
}

locals {
  container_port = "8080"
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/${var.team}/core/terraform/${var.environment}"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}

locals {
  cluster_name = "${data.terraform_remote_state.core.common_cluster_name}"
}

# Resources

resource "aws_iam_role" "gea_task" {
  name               = "${var.ssm_iam_policy_role_name}"
  assume_role_policy = "${data.aws_iam_policy_document.gea_task.json}"
}

data "aws_caller_identity" "current" {}

resource "aws_iam_role_policy" "gea_task_policy" {
  name = "${var.team}-${var.name}-${var.environment}-policy"
  role = "${aws_iam_role.gea_task.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "dynamodb:BatchGetItem",
        "dynamodb:BatchWriteItem",
        "dynamodb:DeleteItem",
        "dynamodb:GetItem",
        "dynamodb:GetRecords",
        "dynamodb:GetShardIterator",
        "dynamodb:PutItem",
        "dynamodb:Query",
        "dynamodb:Scan",
        "dynamodb:UpdateItem"
      ],
      "Resource": [
        "*"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "sqs:ChangeMessageVisibility",
        "sqs:ChangeMessageVisibilityBatch",
        "sqs:DeleteMessage",
        "sqs:DeleteMessageBatch",
        "sqs:ReceiveMessage",
        "sqs:SendMessage",
        "sqs:SendMessageBatch"
      ],
      "Resource": [
        "*"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "sns:Publish"
      ],
      "Resource": [
        "*"
      ]
    }, {
      "Effect": "Allow",
      "Action": [
        "s3:GetObject*",
        "s3:DeleteObject*",
        "s3:PutObject*"
      ],
      "Resource": [
        "*"
      ]
    }
  ]
}
EOF
}

data "aws_iam_policy_document" "gea_task" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_role_policy_attachment" "gea_role_to_sandstorm_policy" {
  role       = "${aws_iam_role.gea_task.name}"
  policy_arn = "${var.sandstorm_policy_arn}"
}

module "simpleservice" {
  source           = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//simpleservice?ref=622157fac36fb40cedcdf9e0b40ca6e5fa1b9917"
  account          = "${var.account}"
  service          = "${var.name}"
  cluster_name     = "${local.cluster_name}"
  region           = "${var.region}"
  team             = "${var.team}"
  environment      = "${var.environment}"
  desired_count    = "${var.desired_count}"
  target_group_arn = "${var.alb_target_group_arn}"
  task_role_arn    = "${aws_iam_role.gea_task.arn}"
  container_port   = "${local.container_port}"
}

module "taskcpuscale" {
  source       = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//taskcpuscale?ref=622157fac36fb40cedcdf9e0b40ca6e5fa1b9917"
  max_size     = "${var.max_size}"
  min_size     = "${var.min_size}"
  account      = "${var.account}"
  region       = "${var.region}"
  cluster_name = "${local.cluster_name}"
  service      = "${module.simpleservice.service_name}"
  team         = "${var.team}"
  environment  = "${var.environment}"
}

output "task_iam_role_arn" {
  value = "${aws_iam_role.gea_task.arn}"
}

module "pipeline" {
  source          = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//pipeline?ref=622157fac36fb40cedcdf9e0b40ca6e5fa1b9917"
  team            = "${var.team}"
  environment     = "${var.environment}"
  aws_creds       = "${data.terraform_remote_state.core.jenkins_build_secret}"
  profile         = "${var.account}"
  promote_from    = "${var.promote_from}"
  auto_promote    = "${var.auto_promote}"
  service         = "${var.name}"
  deploy_aws_role = "${data.terraform_remote_state.core.build_role}"
  region          = "${var.region}"
  service_name    = "${module.simpleservice.service_name}"
  cluster_name    = "${local.cluster_name}"

  data = "${map(
    "task_role_arn", "${aws_iam_role.gea_task.arn}",
    "container_port", "${local.container_port}",
    "task_name", "${var.name}",
    "task_cpu", "${var.task_cpu}",
    "task_mem", "${var.task_mem}",
    "container_loggroup", "${data.terraform_remote_state.core.container_loggroup}",
    "container_loggroup_region", "${data.terraform_remote_state.core.container_loggroup_region}",
    "task_family", "${module.simpleservice.task_family}"
  )}"
}

output "task_iam_role_name" {
  value = "${aws_iam_role.gea_task.name}"
}
