# Module for provisioning the hard delete user service infra pieces required for the Hard Delete User Worker

# Inputs

## Environment
variable team {}
variable name {}
variable environment {}

## Service Properties
variable hard_delete_user_sns_arn {}

# Resources

// hard delete user SQS queue
resource "aws_sqs_queue" "hard_delete_user" {
  name                       = "${var.team}-${var.environment}-hard-delete-user"
  visibility_timeout_seconds = 900

  redrive_policy = <<EOF
  {
    "deadLetterTargetArn": "${aws_sqs_queue.hard_delete_user_dead_letter_queue.arn}",
    "maxReceiveCount": 3
  }
EOF
}

// Add permissions to the hard delete user SQS queue
resource "aws_sqs_queue_policy" "hard_delete_user_permissions" {
  queue_url = "${aws_sqs_queue.hard_delete_user.id}"

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "sqspolicy",
  "Statement": [
    {
      "Sid": "First",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.hard_delete_user.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${var.hard_delete_user_sns_arn}"
        }
      }
    }
  ]
}
POLICY
}

// Dead letter SQS queue that recieves messages from the hard delete user queue that could not be processed.
resource "aws_sqs_queue" "hard_delete_user_dead_letter_queue" {
  name = "${var.team}-${var.environment}-hard-delete-user-deadletter"
}

# Add the subscription to hard delete user SNS topic
resource "aws_sns_topic_subscription" "hard_delete_user_sqs_target" {
  topic_arn = "${var.hard_delete_user_sns_arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.hard_delete_user.arn}"
}

# Consul key that contains the URL of the hard delete user SQS queue, for services to reference
module "hard_delete_user_queue_consulconf" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.9.0"
  service     = "${var.name}"
  team        = "${var.team}"
  environment = "${var.environment}"
  key         = "hard_delete_user.sqssource.queue_url"
  value       = "${aws_sqs_queue.hard_delete_user.id}"
}

# Output the URL of the Hard Delete User SQS queue.  Intended to allow Gea's Canary to get the Production queue URL.
output hard_delete_user_sqs_queue_url {
  value = "${aws_sqs_queue.hard_delete_user.id}"
}
