# Creates a scheduled ECS task that periodically sends a POST request to an endpoint.

variable team {}
variable environment {}

# A short, dash separated identifier that is used in generating names for the Cloudwatch event rule, ECS task and other
# related resources.
variable name {}

# A CloudWatch rate expression that defines how often the lambda function should be run.
# e.g. "rate(2 minutes)"
# Rate expression syntax: https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html
variable rate {}

# The URL of the endpoint that the lambda should send a POST request to.
variable gea_endpoint {}

variable ecs_cluster_id {}

locals {
  image = "docker-registry.internal.justin.tv/events/poll-endpoint:4fa5e18fbbf2de8bfe65430afb6902719c7c5162"
}

# AWS Cloudwatch event rule that fires periodically.
resource "aws_cloudwatch_event_rule" "poller" {
  name                = "${var.team}-${var.environment}-${var.name}"
  description         = "Periodically triggers ECS task to send an HTTP request."
  schedule_expression = "${var.rate}"
}

# Configure the Cloudwatch event rule to run the ECS task when triggered.
resource "aws_cloudwatch_event_target" "poller" {
  rule      = "${aws_cloudwatch_event_rule.poller.name}"
  target_id = "${var.team}-${var.environment}-${var.name}"
  arn       = "${var.ecs_cluster_id}"

  # CloudWatch Events makes calls to Amazon ECS to run tasks on our behalf.  Configure it to use the built-in
  # ecsEventsRole role to grant it the permissions to do so.
  role_arn = "${data.aws_iam_role.ecs_events_role.arn}"

  ecs_target {
    task_count          = 1
    task_definition_arn = "${aws_ecs_task_definition.poller.arn}"
  }
}

# Define the ECS task.
resource "aws_ecs_task_definition" "poller" {
  family                = "${var.team}-${var.environment}-${var.name}"
  container_definitions = "${data.template_file.container_definitions.rendered}"
  network_mode          = "bridge"
}

data "template_file" "container_definitions" {
  template = "${file("${path.module}/container_definitions.tpl")}"

  vars {
    team        = "${var.team}"
    environment = "${var.environment}"
    name        = "${var.name}"

    image    = "${local.image}"
    endpoint = "${var.gea_endpoint}"
  }
}

# The ecsEventsRole IAM role is an automatically created role that has an IAM policy that grants permission to
# run ECS tasks.
data "aws_iam_role" "ecs_events_role" {
  name = "ecsEventsRole"
}
