variable "team" {}
variable "name" {}
variable "environment" {}
variable "rds_multi_az" {}
variable "owner" {}
variable "team_code" {}
variable "region" {}
variable "account" {}
variable "allocated_storage" {}
variable "period" {}
variable "threshold_percent" {}
variable "rds_skip_final_snapshot" {}
variable "twitch_subnets_sg" {}
variable "action_arn" {}

resource "aws_db_instance" "rds_master" {
  identifier = "${var.team_code}-${var.name}-${var.environment}"

  allocated_storage = "${var.allocated_storage}"
  engine            = "postgres"
  engine_version    = "9.6.11"
  instance_class    = "db.m4.large"
  name              = "${var.name}"

  username = "creatorbusiness"
  password = "temporary"

  lifecycle {
    ignore_changes = ["password"]
  }

  backup_retention_period = 1
  backup_window           = "02:00-04:00"
  multi_az                = "${var.rds_multi_az}"
  port                    = 5432
  publicly_accessible     = false
  skip_final_snapshot     = "${var.rds_skip_final_snapshot}"

  vpc_security_group_ids = [
    "${var.twitch_subnets_sg}",
  ]

  db_subnet_group_name = "twitch_subnets"

  tags {
    Name    = "${var.name}"
    Owner   = "${var.owner}"
    Project = "${var.team_code}/${var.name}"
    Team    = "${var.team}"
  }
}

# Create an alarm for when we are above 80% capacity on RDS
resource "aws_cloudwatch_metric_alarm" "rds_disk_utilization_alarm" {
  alarm_name          = "${var.team_code}-${var.name}-${var.environment}-rds-disk-utilization"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods  = 5
  metric_name         = "FreeStorageSpace"
  namespace           = "AWS/RDS"
  period              = "${var.period}"
  statistic           = "Minimum"
  alarm_actions       = ["${var.action_arn}"]

  # Threshold of bytes below which the FreeStorageSpace has to dip for the alarm to fire
  # Currently [var.allocated_storage (in GB) * 1024 (bytes -> kb) * 1024 (kb -> mb) * 1024 (mb -> gb) * percentage threshold]
  threshold = "${var.allocated_storage * 1024 * 1024 * 1024 * (100 - var.threshold_percent) / 100}"

  alarm_description = "Alerts when FreeStorageSpace is below ${100 - var.threshold_percent}% for ${var.team_code}-${var.name}-${var.environment} for ${var.period} seconds"

  dimensions {
    DBInstanceIdentifier = "${var.team_code}-${var.name}-${var.environment}"
  }
}
