# Module for provisioning the upload service infra pieces required for the Generic Upload Service

# Inputs

## Environment
variable "name" {}

variable "owner" {}
variable team {}

variable "environment" {}

## Service Properties
variable "upload_service_arns" {
  type = "list"
}

# Resources

resource "aws_s3_bucket" "upload_storage" {
  bucket = "${var.team}-${var.name}-${var.environment}-upload-storage"
  acl    = "private"

  tags {
    Owner = "${var.owner}"
  }

  lifecycle_rule {
    id      = "expire-uploaded-images"
    prefix  = "uploaded/"
    enabled = true

    expiration {
      days                         = 90
      expired_object_delete_marker = true
    }
  }
}

resource "aws_sns_topic" "upload_topic" {
  name = "${var.team}-${var.name}-${var.environment}-upload-topic"
}

resource "aws_sqs_queue" "upload_deadletter_jobs" {
  name = "${var.team}-${var.name}-${var.environment}-upload-jobs-deadletter"
}

resource "aws_sqs_queue" "upload_jobs" {
  name                      = "${var.team}-${var.name}-${var.environment}-upload-jobs"
  message_retention_seconds = 345600

  redrive_policy = <<EOF
  {
    "deadLetterTargetArn": "${aws_sqs_queue.upload_deadletter_jobs.arn}",
    "maxReceiveCount": 3
  }
EOF
}

data "aws_iam_policy_document" "sqs_policy" {
  statement {
    effect = "Allow"

    principals {
      type        = "AWS"
      identifiers = ["*"]
    }

    actions   = ["sqs:SendMessage"]
    resources = ["${aws_sqs_queue.upload_jobs.arn}"]

    condition = {
      test     = "ArnEquals"
      variable = "aws:SourceArn"
      values   = ["${aws_sns_topic.upload_topic.arn}"]
    }
  }
}

resource "aws_sqs_queue_policy" "sqs_policy" {
  queue_url = "${aws_sqs_queue.upload_jobs.id}"
  policy    = "${data.aws_iam_policy_document.sqs_policy.json}"
}

resource "aws_sns_topic_subscription" "sqs_sns_sub" {
  topic_arn = "${aws_sns_topic.upload_topic.arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.upload_jobs.arn}"
}

data "aws_iam_policy_document" "sns_policy" {
  statement {
    effect = "Allow"

    principals {
      type        = "AWS"
      identifiers = ["${var.upload_service_arns}"]
    }

    actions   = ["sns:Publish"]
    resources = ["${aws_sns_topic.upload_topic.arn}"]
  }
}

resource "aws_sns_topic_policy" "sns_policy" {
  arn    = "${aws_sns_topic.upload_topic.arn}"
  policy = "${data.aws_iam_policy_document.sns_policy.json}"
}

# Output the URL of the Upload SQS queue.  Intended to allow Gea's Canary to get the Production queue URL.
output uploads_sqs_queue_url {
  value = "${aws_sqs_queue.upload_jobs.id}"
}

# Output the ARN of the Upload SNS topic.  Intended to allow Gea's Canary to get the Production SNS topic ARN.
output uploads_sns_topic_arn {
  value = "${aws_sns_topic.upload_topic.arn}"
}

# Output the bucket name of the Upload S3 bucket.  Intended to allow Gea's Canary to get the Production bucket name.
output uploads_s3_bucket {
  value = "${aws_s3_bucket.upload_storage.bucket}"
}

# Output the Upload S3 bucket ID.  Intended to allow another module to attach permissions to the bucket.
output uploads_s3_bucket_id {
  value = "${aws_s3_bucket.upload_storage.id}"
}

# Output the Upload S3 bucket ARN.  Intended to allow another module to attach permissions to the bucket.
output uploads_s3_bucket_arn {
  value = "${aws_s3_bucket.upload_storage.arn}"
}
