package hallpass

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type deleteV1EditorSuite struct {
	suite.Suite
}

func TestDeleteV1EditorSuite(t *testing.T) {
	suite.Run(t, &deleteV1EditorSuite{})
}

func (s *deleteV1EditorSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.DeleteV1Editor(context.Background(), "999999", "111111", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *deleteV1EditorSuite) TestMalformedResponse() {
	payload := `
        {
            "status": 200,
            "message": "malformed JSON String with extra double-quotes""
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.DeleteV1Editor(context.Background(), "999999", "111111", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *deleteV1EditorSuite) TestSuccess() {
	channelID := "999999"
	editorID := "111111"

	payload := `
        {
            "status": "success"
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/permissions/channels/%s/editors/%s", channelID, editorID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.DeleteV1Editor(context.Background(), "999999", "111111", nil)
	s.NoError(err)

	if s.NotNil(resp) {
		s.Equal(resp.Status, "success")
	}
}
