package hallpass

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type getV1EditorsSuite struct {
	suite.Suite
}

func TestGetV1EditorsSuite(t *testing.T) {
	suite.Run(t, &getV1EditorsSuite{})
}

func (s *getV1EditorsSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1Editors(context.Background(), "999999", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *getV1EditorsSuite) TestMalformedResponse() {
	payload := `
        {
            "status": 200,
            "message": "malformed JSON String with extra double-quotes""
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1Editors(context.Background(), "999999", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *getV1EditorsSuite) TestSuccess() {
	channelID := "999999"

	payload := `
        {
            "editors": [
                "000000",
                "111111",
                "222222"
            ]
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/permissions/channels/%s/editors", channelID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1Editors(context.Background(), channelID, nil)
	s.NoError(err)

	if s.NotNil(resp) {
		s.Equal(resp.Editors[0], "000000")
		s.Equal(resp.Editors[1], "111111")
		s.Equal(resp.Editors[2], "222222")

	}
}
