package hallpass

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type getV1IsEditorSuite struct {
	suite.Suite
}

func TestGetV1IsEditorSuite(t *testing.T) {
	suite.Run(t, &getV1IsEditorSuite{})
}

func (s *getV1IsEditorSuite) TestFailure() {
	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1IsEditor(context.Background(), "999999", "111111", nil)
	s.NotNil(err)
	s.Nil(resp)
}

func (s *getV1IsEditorSuite) TestMalformedResponse() {
	payload := `
        {
            "status": 200,
            "message": "malformed JSON String with extra double-quotes""
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1IsEditor(context.Background(), "999999", "111111", nil)
	s.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	s.NoError(err)
	s.NotEqual(string(jsonResponse), payload)
}

func (s *getV1IsEditorSuite) TestSuccess() {
	channelID := "999999"
	editorID := "111111"

	payload := `
        {
            "is_editor": true
        }
    `

	handler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		s.Equal(fmt.Sprintf("/v1/permissions/channels/%s/editors/%s", channelID, editorID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		s.Require().NoError(err)
	})

	server := httptest.NewServer(handler)

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	s.Require().NoError(err)

	resp, err := client.GetV1IsEditor(context.Background(), "999999", "111111", nil)
	s.NoError(err)

	if s.NotNil(resp) {
		s.Equal(resp.IsEditor, true)

	}
}
