package main

import (
	"context"
	"net/http"
	"os"

	"code.justin.tv/cb/hallpass/internal/cache"
	"code.justin.tv/cb/hallpass/internal/db"
	"code.justin.tv/cb/hallpass/internal/httputil"
	"code.justin.tv/cb/hallpass/internal/logging"
	"code.justin.tv/cb/hallpass/internal/statsd"
	"code.justin.tv/cb/hallpass/internal/worker"

	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

func init() {
	logging.SetupRollbar(os.Getenv("ROLLBAR_TOKEN"), os.Getenv("ENVIRONMENT"))
}

func main() {
	dbWriter, err := db.NewClient(db.ClientConfig{
		Host:     os.Getenv("DB_MASTER_HOST"),
		Port:     os.Getenv("DB_MASTER_PORT"),
		Name:     os.Getenv("DB_NAME"),
		User:     os.Getenv("DB_USER"),
		Password: os.Getenv("DB_PASSWORD"),
		SSLMode:  os.Getenv("DB_SSLMODE"),
	})
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new db client for master connection")
		return
	}

	redisClient := cache.NewRedisClient(cache.RedisClientConfig{
		Addr:     os.Getenv("REDIS_HOST"),
		Password: "",
		DB:       0,
	})

	statsdClient, err := statsd.NewClient(os.Getenv("STATSD_HOST"), os.Getenv("ENVIRONMENT"))
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate statsd client")
		return
	}

	server := &http.Server{
		Addr: ":8000",
		Handler: worker.NewServer(&worker.ServerParams{
			DBWriter:    dbWriter,
			RedisClient: redisClient,
			Statsd:      statsdClient,
		}),
	}

	go func() {
		log.Info("server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("server failed fatally while listening")
		}
	}()

	httputil.Graceful(context.Background(), server)
}
