package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/hallpass/internal/mocks"
	"code.justin.tv/cb/hallpass/internal/statsd"
	"code.justin.tv/cb/hallpass/view"

	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type getV1EditorsSuite struct {
	suite.Suite
	server       *Server
	dbReader     *mocks.PermissionsDB
	dbWriter     *mocks.PermissionsDB
	redisClient  *mocks.PermissionsCache
	usersService *mocks.UsersServiceClient
	statsd       *statsd.NoopClient
	ChannelID    string
	Path         string
}

func TestGetV1EditorsSuite(t *testing.T) {
	suite.Run(t, &getV1EditorsSuite{})
}

func (s *getV1EditorsSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)
	s.dbReader = &mocks.PermissionsDB{}
	s.dbWriter = &mocks.PermissionsDB{}
	s.redisClient = &mocks.PermissionsCache{}
	s.usersService = &mocks.UsersServiceClient{}
	s.statsd = statsd.NewNoopClient()

	params := &ServerParams{
		DBReader:     s.dbReader,
		DBWriter:     s.dbWriter,
		RedisClient:  s.redisClient,
		UsersService: s.usersService,
		Statsd:       s.statsd,
	}

	s.server = NewServer(params)

	s.ChannelID = "123456"
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editors", s.ChannelID)
}

func (s *getV1EditorsSuite) TestWithQueryParam_CacheFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, errors.New("failed to query editors table"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV1EditorsSuite) TestWithQueryParam_DBFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, errors.New("failed to query editors table"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV1EditorsSuite) TestWithQueryParam_SuccessWithEditors() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, nil)
	s.redisClient.On("CacheEditors", s.ChannelID, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{"654321", "999999"}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	response := view.GetEditorsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.Editors[0], "654321")
	s.Equal(response.Editors[1], "999999")
}

func (s *getV1EditorsSuite) TestWithQueryParam_SuccessWithNoEditors() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, nil)
	s.redisClient.On("CacheEditors", s.ChannelID, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	response := view.GetEditorsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(len(response.Editors), 0)
}

func (s *getV1EditorsSuite) TestWithQueryParam_SuccessWithEditorsFromCache() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{"654321", "999999"}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertNotCalled(s.T(), "GetEditors", mock.Anything, s.ChannelID)
	s.redisClient.AssertExpectations(s.T())

	response := view.GetEditorsResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.Editors[0], "654321")
	s.Equal(response.Editors[1], "999999")
}
