package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/hallpass/internal/mocks"
	"code.justin.tv/cb/hallpass/internal/statsd"
	"code.justin.tv/cb/hallpass/view"

	log "github.com/sirupsen/logrus"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type getV1IsEditorsSuite struct {
	suite.Suite
	server       *Server
	dbReader     *mocks.PermissionsDB
	dbWriter     *mocks.PermissionsDB
	redisClient  *mocks.PermissionsCache
	usersService *mocks.UsersServiceClient
	statsd       *statsd.NoopClient
	ChannelID    string
	EditorID     string
	Path         string
}

func TestGetV1IsEditorsSuite(t *testing.T) {
	suite.Run(t, &getV1IsEditorsSuite{})
}

func (s *getV1IsEditorsSuite) SetupTest() {
	log.SetLevel(log.PanicLevel)
	s.dbReader = &mocks.PermissionsDB{}
	s.dbWriter = &mocks.PermissionsDB{}
	s.redisClient = &mocks.PermissionsCache{}
	s.usersService = &mocks.UsersServiceClient{}
	s.statsd = statsd.NewNoopClient()

	params := &ServerParams{
		DBReader:     s.dbReader,
		DBWriter:     s.dbWriter,
		RedisClient:  s.redisClient,
		UsersService: s.usersService,
		Statsd:       s.statsd,
	}

	s.server = NewServer(params)

	s.ChannelID = "123456"
	s.EditorID = "999999"
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editors/%s", s.ChannelID, s.EditorID)
}

func (s *getV1IsEditorsSuite) TestCacheFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, errors.New("failed to query permissions table"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV1IsEditorsSuite) TestBadRequest_ChannelID() {
	recorder := httptest.NewRecorder()
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editors/%s", "bleh", s.EditorID)
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusBadRequest, recorder.Code)
}

func (s *getV1IsEditorsSuite) TestBadRequest_EditorID() {
	recorder := httptest.NewRecorder()
	s.Path = fmt.Sprintf("/v1/permissions/channels/%s/editors/%s", s.ChannelID, "bleh")
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusBadRequest, recorder.Code)
}

func (s *getV1IsEditorsSuite) TestDBFailure() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, errors.New("failed to query permissions table"))

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	s.Equal(http.StatusInternalServerError, recorder.Code)
}

func (s *getV1IsEditorsSuite) TestSuccessWithEditor() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, nil)
	s.redisClient.On("CacheEditors", s.ChannelID, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{"654321", "999999"}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	response := view.GetIsEditorResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.IsEditor, true)
}

func (s *getV1IsEditorsSuite) TestSuccessWithNoEditor() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{}, nil)
	s.redisClient.On("CacheEditors", s.ChannelID, mock.Anything, mock.Anything).Return(nil)
	s.dbReader.On("GetEditors", mock.Anything, s.ChannelID).Return([]string{}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertExpectations(s.T())
	s.redisClient.AssertExpectations(s.T())

	response := view.GetIsEditorResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.IsEditor, false)
}

func (s *getV1IsEditorsSuite) TestSuccessWithEditorFromCache() {
	recorder := httptest.NewRecorder()
	req, err := http.NewRequest(http.MethodGet, s.Path, nil)
	s.Require().NoError(err)

	s.redisClient.On("GetCachedEditors", s.ChannelID).Return([]string{"654321", "999999"}, nil)

	s.server.ServeHTTP(recorder, req)

	s.dbReader.AssertNotCalled(s.T(), "GetEditors", mock.Anything, s.ChannelID)
	s.redisClient.AssertExpectations(s.T())
	response := view.GetIsEditorResponse{}
	err = json.Unmarshal(recorder.Body.Bytes(), &response)
	s.NoError(err)

	s.Equal(http.StatusOK, recorder.Code)
	s.Equal(response.IsEditor, true)
}
