package api

import (
	"encoding/json"
	"errors"
	"net/http"

	"goji.io/pat"

	"code.justin.tv/cb/hallpass/internal/httputil"
	"code.justin.tv/cb/hallpass/view"
)

func (s *Server) postV1Editor(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	grantedFor := pat.Param(req, "channel_id")

	if req.Body == nil {
		jsonWriter.BadRequest("request body is required")
		return
	}

	decoder := json.NewDecoder(req.Body)
	reqBody := &view.CreateEditorRequest{}
	err := decoder.Decode(reqBody)

	if err != nil {
		jsonWriter.BadRequest("could not read request body")
		return
	}

	err = validateEditorReqBody(reqBody)
	if err != nil {
		jsonWriter.BadRequest(err)
		return
	}

	_, err = s.validateUser(req.Context(), jsonWriter, reqBody.GrantedBy)
	if err != nil {
		return
	}

	_, err = s.validateUser(req.Context(), jsonWriter, grantedFor)
	if err != nil {
		return
	}

	_, err = s.validateUser(req.Context(), jsonWriter, reqBody.GrantedTo)
	if err != nil {
		return
	}

	// Get list of current permissions and ensure we aren't adding a duplicate
	currEditors, err := s.dbReader.GetEditors(req.Context(), grantedFor)
	if err != nil {
		jsonWriter.InternalServerError("failed to look up current editors to check for duplicates", err)
		return
	}

	for _, editor := range currEditors {
		if editor == reqBody.GrantedTo {
			jsonWriter.UnprocessableEntity("The requested channel id is already an editor")
			return
		}
	}

	err = s.dbWriter.CreateEditor(req.Context(), grantedFor, reqBody.GrantedTo)
	if err != nil {
		jsonWriter.InternalServerError("failed to create editor in editors table", err)
		return
	}

	// Invalidate the cache here
	s.redisClient.RemoveAllEditors(grantedFor)
	s.redisClient.RemoveAllEditable(reqBody.GrantedTo)

	response := view.CreateEditorResponse{
		GrantedTo: reqBody.GrantedTo,
	}

	jsonWriter.OK(response)
}

func validateEditorReqBody(reqBody *view.CreateEditorRequest) error {
	if reqBody.GrantedBy == "" {
		return errors.New("ID of granted for user is required")
	}

	if reqBody.GrantedTo == "" {
		return errors.New("ID of granted to user is required")
	}

	return nil
}
