package api

import (
	"net/http"

	"code.justin.tv/cb/hallpass/internal/httputil"

	"goji.io"
	"goji.io/pat"
)

// Server contains a router and client interfaces to downstream services.
type Server struct {
	mux          *goji.Mux
	dbReader     PermissionsDB
	dbWriter     PermissionsDB
	redisClient  PermissionsCache
	usersService UsersServiceClient
	statsd       StatsdClient
}

// ServerParams contains the required params for creating a server.
type ServerParams struct {
	DBReader     PermissionsDB
	DBWriter     PermissionsDB
	RedisClient  PermissionsCache
	UsersService UsersServiceClient
	Statsd       StatsdClient
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
// client.go in the db package implements the PermissionsDB interface. We pass in two
// separate clients, one for the master db and one for the replica db.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:          goji.NewMux(),
		dbReader:     params.DBReader,
		dbWriter:     params.DBWriter,
		redisClient:  params.RedisClient,
		usersService: params.UsersService,
		statsd:       params.Statsd,
	}

	server.mux.Use(httputil.PanicRecoveryMiddleware)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.AccessLogMiddleware)
	root.Use(server.statsd.StatsLogger)

	v1 := goji.SubMux()
	root.Handle(pat.New("/v1/*"), v1)

	channels := goji.SubMux()
	v1.Handle(pat.New("/permissions/channels/:channel_id/*"), channels)
	channels.Use(validateNumericChannelID)

	channels.HandleFunc(pat.Get("/editors"), server.getV1Editors)
	channels.HandleFunc(pat.Get("/editable_channels"), server.getV1EditableChannels)
	channels.HandleFunc(pat.Post("/editors"), server.postV1Editor)

	editors := goji.SubMux()
	channels.Handle(pat.New("/editors/:editor_id"), editors)
	editors.Use(validateNumericEditorID)

	editors.HandleFunc(pat.Get(""), server.getV1IsEditor)
	editors.HandleFunc(pat.Delete(""), server.deleteV1Editor)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
