package cache

const (
	// IsEditorRedisField is the name of the map field in the redis cache for the isEditor value
	IsEditorRedisField = "isEditor"

	// GrantedByRedisField is the name of the map field in the redis cache for the grantedBy value
	GrantedByRedisField = "grantedBy"

	editorsKey = "editors"

	editableKey = "editable"
)

// GetCachedEditors returns any editors in the cache for the given channel id. If no existing editors
// exist, an empty array is returned.
func (c *RedisClient) GetCachedEditors(channelID string) ([]string, error) {
	key := buildRedisKey(channelID, editorsKey)
	cachedEditors, err := c.cache.SMembers(key).Result()
	if err != nil {
		return nil, err
	}

	return cachedEditors, nil
}

// GetCachedEditable returns any editable channels in the cache for the given channel id.
func (c *RedisClient) GetCachedEditable(channelID string) ([]string, error) {
	key := buildRedisKey(channelID, editableKey)
	cachedEditable, err := c.cache.SMembers(key).Result()
	if err != nil {
		return nil, err
	}

	return cachedEditable, nil
}

// CacheEditors stores a editor list in the cache.
func (c *RedisClient) CacheEditors(channelID string, editorIDs []string) {
	if len(editorIDs) == 0 {
		return
	}

	key := buildRedisKey(channelID, editorsKey)

	mEditors := make([]interface{}, len(editorIDs))
	for i, editor := range editorIDs {
		mEditors[i] = editor
	}

	c.cache.SAdd(key, mEditors...)
}

// CacheEditable stores a editable list in the cache.
func (c *RedisClient) CacheEditable(channelID string, editableIDs []string) {
	if len(editableIDs) == 0 {
		return
	}

	key := buildRedisKey(channelID, editableKey)

	mEditable := make([]interface{}, len(editableIDs))
	for i, edit := range editableIDs {
		mEditable[i] = edit
	}

	c.cache.SAdd(key, mEditable...)
}

// RemoveEditor deletes a single existing editor from the cache.
func (c *RedisClient) RemoveEditor(channelID string, editorID string) {
	key := buildRedisKey(channelID, editorsKey)
	c.cache.SRem(key, editorID)
}

// RemoveEditable deletes a single existing editable from the cache.
func (c *RedisClient) RemoveEditable(channelID string, editableID string) {
	key := buildRedisKey(channelID, editableKey)
	c.cache.SRem(key, editableID)
}

// RemoveAllEditors deletes all existing editors from the cache.
func (c *RedisClient) RemoveAllEditors(channelID string) {
	key := buildRedisKey(channelID, editorsKey)
	c.cache.Del(key)
}

// RemoveAllEditable deletes all existing editable channels from the cache.
func (c *RedisClient) RemoveAllEditable(channelID string) {
	key := buildRedisKey(channelID, editableKey)
	c.cache.Del(key)
}

func buildRedisKey(channelID string, postfix string) string {
	return channelID + "_" + postfix
}
