package db

import (
	"context"
	"database/sql"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// GetEditors returns a list of editors for a given channel id.
func (c *Client) GetEditors(ctx context.Context, channelID string) ([]string, error) {
	query := `
        SELECT channel_id,
               editor_user_id
        FROM   editors
        WHERE  channel_id = $1
    `

	rows, err := c.db.QueryContext(ctx, query, channelID)

	switch {
	case err == sql.ErrNoRows:
		return []string{}, nil
	case err != nil:
		return nil, errors.Wrap(err, "db: failed to scan editors row")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":  query,
				"channel_id": channelID,
			}).Error("db: failed to close rows")
		}
	}()

	return convertRowsToEditors(rows)
}

// GetEditableChannels returns a list of channels that a given channelID can edit
func (c *Client) GetEditableChannels(ctx context.Context, channelID string) ([]string, error) {
	query := `
        SELECT channel_id,
               editor_user_id
        FROM   editors
        WHERE  editor_user_id = $1
    `

	editors := []string{}
	rows, err := c.db.QueryContext(ctx, query, channelID)

	switch {
	case err == sql.ErrNoRows:
		return editors, nil
	case err != nil:
		return nil, errors.Wrap(err, "db: failed to scan editors row")
	}

	defer func() {
		err = rows.Close()
		if err != nil {
			log.WithError(err).WithFields(log.Fields{
				"statement":  query,
				"channel_id": channelID,
			}).Error("db: failed to close rows")
		}
	}()

	return convertRowsToEditable(rows)
}

// CreateEditor grants a new editor permission for a user
func (c *Client) CreateEditor(ctx context.Context, channelID string, editorUserID string) error {
	query := `
        INSERT INTO editors
                    (channel_id, editor_user_id)
        VALUES      ($1, $2)
    `

	_, err := c.db.ExecContext(ctx, query, channelID, editorUserID)

	if err != nil {
		return errors.Wrap(err, "db: failed to insert new permission")
	}

	return nil
}

// DeleteEditor removes a single editing permission for a user and an existing editor
func (c *Client) DeleteEditor(ctx context.Context, channelID string, editorID string) error {
	query := `
        DELETE FROM editors
        WHERE       channel_id = $1
        AND         editor_user_id = $2
    `

	_, err := c.db.ExecContext(ctx, query, channelID, editorID)

	if err != nil {
		return errors.Wrap(err, "db: failed to delete permission")
	}

	return nil
}

// DeleteAllEditors removes all editing permissions for a user
func (c *Client) DeleteAllEditors(ctx context.Context, channelID string) error {
	query := `
		DELETE FROM editors
		WHERE channel_id = $1
	`

	_, err := c.db.ExecContext(ctx, query, channelID)

	if err != nil {
		return errors.Wrap(err, "db: failed to delete all editor permissions")
	}

	return nil
}

func convertRowsToEditors(rows *sql.Rows) ([]string, error) {
	editors := []string{}

	for rows.Next() {
		permission := EditorPermission{}

		scanErr := rows.Scan(&permission.ChannelID, &permission.EditorUserID)
		if scanErr != nil {
			return nil, errors.Wrap(scanErr, "db: failed to scan editors rows")
		}

		editors = append(editors, permission.EditorUserID)
	}

	err := rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate editors rows")
	}

	return editors, nil
}

func convertRowsToEditable(rows *sql.Rows) ([]string, error) {
	editable := []string{}

	for rows.Next() {
		permission := EditorPermission{}

		scanErr := rows.Scan(&permission.ChannelID, &permission.EditorUserID)
		if scanErr != nil {
			return nil, errors.Wrap(scanErr, "db: failed to scan editable rows")
		}

		editable = append(editable, permission.ChannelID)
	}

	err := rows.Err()
	if err != nil {
		return nil, errors.Wrap(err, "db: failed to iterate editable rows")
	}

	return editable, nil
}
