package statsd

import (
	"net/http"
	"strconv"
	"strings"
	"time"

	log "github.com/sirupsen/logrus"
)

type responseWriter struct {
	http.ResponseWriter
	status int
}

const sampleRate = 1.0

// StatsLogger is an HTTP middleware that logs the request.
func (c *Client) StatsLogger(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		start := time.Now().UTC()
		logged := &responseWriter{
			ResponseWriter: w,
		}

		inner.ServeHTTP(logged, r)

		duration := time.Since(start)
		statName := buildStatName(r)

		go func() {
			err := c.TimingDuration(statName, duration, sampleRate)
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"stat_name": statName,
					"duration":  duration,
				}).Warn("statsd: failed to send timing duration")
			}

			err = c.Inc(statName, 1, sampleRate)
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"stat_name": statName,
				}).Warn("statsd: failed to send stat inc")
			}
		}()
	}

	return http.HandlerFunc(middleware)
}

func buildStatName(r *http.Request) string {
	var statNameArr []string
	splitStr := strings.Split(strings.Trim(r.RequestURI, "/"), "/")

	// Strip any numbers that might be id's out of the stat name, since graphite cannot
	// handle them.
	for _, str := range splitStr {
		if _, err := strconv.Atoi(str); err != nil {
			statNameArr = append(statNameArr, str)
		}
	}

	statName := strings.Join(statNameArr, ".")
	statName = statName + ("." + r.Method)

	return statName
}

func (rw *responseWriter) WriteHeader(status int) {
	rw.status = status
	rw.ResponseWriter.WriteHeader(status)
}

// StatsLogger is an HTTP middleware that does nothing when implemented for the noop client.
func (c *NoopClient) StatsLogger(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, r *http.Request) {
		logged := &responseWriter{
			ResponseWriter: w,
		}

		inner.ServeHTTP(logged, r)
	}

	return http.HandlerFunc(middleware)
}
