package worker

import (
	"net/http"

	"code.justin.tv/cb/hallpass/internal/api"
	"code.justin.tv/cb/hallpass/internal/httputil"

	"goji.io"
	"goji.io/pat"
)

// Server contains a router and client interfaces to downstream services.
type Server struct {
	mux         *goji.Mux
	dbWriter    api.PermissionsDB
	redisClient api.PermissionsCache
	statsd      api.StatsdClient
}

// ServerParams contains the required params for creating a server.
type ServerParams struct {
	DBWriter    api.PermissionsDB
	RedisClient api.PermissionsCache
	Statsd      api.StatsdClient
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
// client.go in the db package implements the PermissionsDB interface. We pass in two
// separate clients, one for the master db and one for the replica db.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:         goji.NewMux(),
		dbWriter:    params.DBWriter,
		redisClient: params.RedisClient,
		statsd:      params.Statsd,
	}

	server.mux.Use(httputil.PanicRecoveryMiddleware)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.AccessLogMiddleware)
	root.Use(server.statsd.StatsLogger)

	root.HandleFunc(pat.Post("/process"), server.process)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
