// +build integration

package duplo_test

import (
	"testing"
	"time"

	"math/rand"
	"strconv"

	"sync"

	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func newSetup() *TestSetup {
	return &TestSetup{
		Common: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "client_int_tests",
			BaseDirectory: "../",
		},
	}
}

var r *rand.Rand
var mu sync.Mutex

func init() {
	r = rand.New(rand.NewSource(time.Now().UnixNano()))
}

func newTestUserID() string {
	mu.Lock()
	defer mu.Unlock()
	return strconv.FormatInt(int64(r.Int31()), 10)
}

func timestamp() string {
	return strconv.FormatInt(time.Now().UnixNano(), 10)
}

func TestClientIntegration_Post(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		t.Log("using client addr ", cl.Config.Host())
		testUserID := newTestUserID()

		Convey("Update a post that doesn't exist should fail", func() {
			err := cl.UpdatePost(ctx, strconv.Itoa(int(time.Now().UnixNano())), duplo.UpdatePostOptions{Emotes: &[]duplo.Emote{
				{
					ID:    1,
					Start: 2,
					End:   3,
					Set:   4,
				},
			},
				EmbedURLs: &[]string{}})
			So(err, ShouldNotBeNil)
		})

		Convey("Should be able to create and get a post with embed entity", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			testEntity := entity.New(entity.NamespaceClip, "zzxyzz")
			created, err := cl.CreatePost(ctx, testUserID, testBody, &duplo.CreatePostOptions{
				EmbedEntities: &[]entity.Entity{testEntity},
			})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)
			So(created.EmbedURLs, ShouldBeNil)
			So(created.EmbedEntities, ShouldNotBeNil)
			So(len(*created.EmbedEntities), ShouldEqual, 1)
			So((*created.EmbedEntities)[0].Encode(), ShouldEqual, testEntity.Encode())

			post, err := cl.GetPost(ctx, created.ID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Body, ShouldEqual, testBody)
			So(post.Emotes, ShouldBeNil)
			So(post.EmbedURLs, ShouldBeNil)
			So(post.EmbedEntities, ShouldNotBeNil)
			So(len(*created.EmbedEntities), ShouldEqual, 1)
			So((*created.EmbedEntities)[0].Encode(), ShouldEqual, testEntity.Encode())
		})

		Convey("Should be able to create and get a post", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			created, err := cl.CreatePost(ctx, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)
			So(created.EmbedEntities, ShouldBeNil)

			post, err := cl.GetPost(ctx, created.ID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Body, ShouldEqual, testBody)
			So(post.Emotes, ShouldBeNil)
			So(post.EmbedEntities, ShouldBeNil)

			Convey("post should update with an emote and an embed url", func() {
				testURL := "https://www.twitch.tv"
				err := cl.UpdatePost(ctx, post.ID, duplo.UpdatePostOptions{
					Emotes: &[]duplo.Emote{
						{
							ID:    1,
							Start: 2,
							End:   3,
							Set:   4,
						},
					},
					EmbedURLs: &[]string{testURL},
				})
				So(err, ShouldBeNil)
				post, err := cl.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldNotBeNil)
				So(post.Body, ShouldEqual, testBody)
				So(post.Emotes, ShouldNotBeNil)
				So(post.EmbedEntities, ShouldBeNil)
				So(len(*post.Emotes), ShouldEqual, 1)
				So((*post.Emotes)[0].Start, ShouldEqual, 2)
				So(post.EmbedURLs, ShouldHaveLength, 1)
				So((*post.EmbedURLs)[0], ShouldEqual, testURL)
			})
			Convey("post should update with an embed entity", func() {
				testEntity := entity.New(entity.NamespaceClip, "abcabcaa")
				err := cl.UpdatePost(ctx, post.ID, duplo.UpdatePostOptions{
					EmbedEntities: &[]entity.Entity{testEntity},
				})
				So(err, ShouldBeNil)
				post, err := cl.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldNotBeNil)
				So(post.Body, ShouldEqual, testBody)
				So(post.Emotes, ShouldBeNil)
				So(post.EmbedURLs, ShouldBeNil)
				So(post.EmbedEntities, ShouldHaveLength, 1)
				So((*post.EmbedEntities)[0].Encode(), ShouldEqual, testEntity.Encode())
			})

			Convey("post should update with zero emotes and embed urls", func() {
				err := cl.UpdatePost(ctx, post.ID, duplo.UpdatePostOptions{
					Emotes:    &[]duplo.Emote{},
					EmbedURLs: &[]string{},
				})
				So(err, ShouldBeNil)
				post, err := cl.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldNotBeNil)
				So(post.Body, ShouldEqual, testBody)
				So(post.Emotes, ShouldNotBeNil)
				So(len(*post.Emotes), ShouldEqual, 0)
				So(post.EmbedURLs, ShouldNotBeNil)
				So(len(*post.Emotes), ShouldEqual, 0)
			})

			Convey("post should remove", func() {
				deleted, err := cl.DeletePost(ctx, created.ID, nil)
				So(err, ShouldBeNil)
				So(deleted, ShouldNotBeNil)
				So(deleted.DeletedAt, ShouldNotBeNil)

				post, err = cl.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldBeNil)

				posts, err := cl.GetPosts(ctx, []string{created.ID})
				So(err, ShouldBeNil)
				So(len(posts.Items), ShouldEqual, 0)
			})

			Convey("Post should be fetchable by userID", func() {
				ids, err := tryXTimes(3, func() (interface{}, error) {
					a, b := cl.GetPostIDsByUser(ctx, testUserID, &duplo.GetPostIDsByUserOptions{Limit: 2})
					if b != nil || len(a.PostIDs) == 0 {
						return nil, errors.New("Unable to fetch so far")
					}
					return a, b
				})
				So(err, ShouldBeNil)
				So(len(ids.(*duplo.PaginatedPostIDs).PostIDs), ShouldEqual, 1)
				So(ids.(*duplo.PaginatedPostIDs).PostIDs[0], ShouldEqual, created.ID)
			})
		})

		Convey("Should be create a post with an audrey_id and get the ID back", func() {
			testBody := "TestClientIntegration_Post_Audrey " + timestamp()
			testAudreyID := "test:audrey:" + timestamp()
			created, err := cl.CreatePost(ctx, testUserID, testBody, &duplo.CreatePostOptions{
				AudreyID: testAudreyID,
			})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)

			postInt, err := tryXTimes(3, func() (interface{}, error) {
				return cl.GetPostIDByLegacyID(ctx, testAudreyID)
			})
			So(err, ShouldBeNil)
			postID := postInt.(*duplo.PostID)
			So(postID, ShouldNotBeNil)
			So(postID.ID, ShouldEqual, created.ID)
		})

		Convey("Should be able to get multiple posts", func() {
			testBody1 := "TestClientIntegration_Post " + timestamp()
			c1, err := cl.CreatePost(ctx, testUserID, testBody1, nil)
			So(err, ShouldBeNil)
			So(c1, ShouldNotBeNil)
			So(c1.ID, ShouldNotBeNil)
			So(c1.Body, ShouldEqual, testBody1)

			testBody2 := "TestClientIntegration_Post " + timestamp()
			c2, err := cl.CreatePost(ctx, testUserID, testBody2, nil)
			So(err, ShouldBeNil)
			So(c2, ShouldNotBeNil)
			So(c2.ID, ShouldNotBeNil)
			So(c2.Body, ShouldEqual, testBody2)

			posts, err := cl.GetPosts(ctx, []string{c1.ID, c2.ID})
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(len(posts.Items), ShouldEqual, 2)
			So(c1, ShouldBeIn, posts.Items)
			So(c2, ShouldBeIn, posts.Items)
			Convey("and if one is shared it should be discovered w/ GetPosts", func() {
				c1Ent := entity.New(entity.NamespacePost, c1.ID)
				c1Share, err := cl.CreateShare(ctx, testUserID, c1Ent, nil)
				So(err, ShouldBeNil)
				So(c1Share, ShouldNotBeNil)

				posts2, err := cl.GetPostsWithOptions(ctx, []string{c1.ID, c2.ID}, &duplo.GetPostsOptions{
					Shares:     true,
					ShareUsers: []string{testUserID, testUserID + "1"},
				})
				So(err, ShouldBeNil)
				So(posts2, ShouldNotBeNil)
			})
		})
	})
}

func tryXTimes(count int, f func() (interface{}, error)) (interface{}, error) {
	var a interface{}
	var b error
	for i := 0; i < count; i++ {
		a, b = f()
		if b == nil {
			return a, b
		}
	}
	return a, b
}

func TestClientIntegration_Comment(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		testUserID := newTestUserID()

		Convey("Should be able to create, get, and delete a comment", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := cl.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.ParentEntity, ShouldResemble, parent)
			So(created.Body, ShouldEqual, testBody)

			commentI, err := tryXTimes(3, func() (interface{}, error) {
				return cl.GetComment(ctx, created.ID)
			})
			So(err, ShouldBeNil)
			comment := commentI.(*duplo.Comment)
			So(comment, ShouldNotBeNil)
			So(comment.Body, ShouldEqual, testBody)

			deleted, err := cl.DeleteComment(ctx, created.ID, nil)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)
			So(deleted.DeletedAt, ShouldNotBeNil)

			comment, err = cl.GetComment(ctx, created.ID)
			So(err, ShouldBeNil)
			So(comment, ShouldBeNil)
		})

		Convey("Should be able to edit a comment's needs approval", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			createOptions := &duplo.CreateCommentOptions{
				NeedsApproval: true,
			}

			created, err := cl.CreateComment(ctx, parent, testUserID, testBody, createOptions)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.NeedsApproval, ShouldBeTrue)

			needsApproval := false
			editOptions := &duplo.UpdateCommentParams{
				NeedsApproval: &needsApproval,
			}
			edited, err := cl.UpdateComment(ctx, created.ID, editOptions)
			So(err, ShouldBeNil)
			So(edited, ShouldNotBeNil)
			So(edited.ID, ShouldEqual, created.ID)
			So(edited.NeedsApproval, ShouldBeFalse)
		})

		Convey("Should be able to update a comment's emotes", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := cl.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.Emotes, ShouldBeNil)

			emotes := []duplo.Emote{
				{
					ID:    1,
					Start: 1,
					End:   2,
					Set:   4,
				}, {
					ID:    5,
					Start: 6,
					End:   9,
					Set:   9,
				},
			}
			updatedComment, err := cl.UpdateComment(ctx, created.ID, &duplo.UpdateCommentParams{
				Emotes: &emotes,
			})
			So(err, ShouldBeNil)
			So(updatedComment, ShouldNotBeNil)
			So(updatedComment.ID, ShouldEqual, created.ID)
			So(updatedComment.Emotes, ShouldNotBeNil)
			So(*updatedComment.Emotes, ShouldResemble, emotes)
		})

		Convey("Should be able to delete comments by parent and user ID", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := cl.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			err = cl.DeleteCommentsByParentAndUser(ctx, parent, testUserID)
			So(err, ShouldBeNil)

			comment, err := cl.GetComment(ctx, created.ID)
			So(err, ShouldBeNil)
			So(comment, ShouldBeNil)
		})

		Convey("Should be able to create a comment with an audrey_id and get the ID back", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment_Audrey " + timestamp()
			testAudreyID := "test:audrey:" + timestamp()
			created, err := cl.CreateComment(ctx, parent, testUserID, testBody, &duplo.CreateCommentOptions{
				AudreyID: testAudreyID,
			})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)

			commentID, err := tryXTimes(3, func() (interface{}, error) {
				return cl.GetCommentIDByLegacyID(ctx, testAudreyID)
			})
			So(err, ShouldBeNil)
			So(commentID, ShouldNotBeNil)
			So(commentID.(*duplo.CommentID).ID, ShouldEqual, created.ID)
		})

		Convey("Should be able to get comments and comments summaries by parent entity", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := cl.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.ParentEntity, ShouldResemble, parent)
			So(created.Body, ShouldEqual, testBody)

			comments, err := cl.GetCommentsByParent(ctx, parent, nil)
			So(err, ShouldBeNil)
			So(comments.Items, ShouldNotBeNil)
			So(comments.Items, ShouldContain, created)

			summaries, err := cl.GetCommentsSummariesByParent(ctx, []entity.Entity{parent})
			So(err, ShouldBeNil)
			So(summaries.Items, ShouldNotBeNil)
			So(len(summaries.Items), ShouldEqual, 1)
			So(summaries.Items[0].ParentEntity, ShouldResemble, parent)
			So(summaries.Items[0].Total, ShouldEqual, 1)
		})
	})
}

func TestClientIntegration_Share(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		testUserID := newTestUserID()
		testEntityID := entity.New("test_post", newTestUserID())

		Convey("Should be able to create a share", func() {
			share, err := cl.CreateShare(ctx, testUserID, testEntityID, nil)
			So(err, ShouldBeNil)
			So(share.ID, ShouldNotEqual, "")
			Convey("and fetch it back", func() {
				share2, err := cl.GetShare(ctx, share.ID)
				So(err, ShouldBeNil)
				So(share2.TargetEntity, ShouldResemble, testEntityID)
			})
			Convey("and fetch it back by entity", func() {
				shares, err := cl.GetSharesByAuthor(ctx, testUserID, []entity.Entity{testEntityID})
				So(err, ShouldBeNil)
				So(shares.Items[0], ShouldNotBeNil)
				So(shares.Items[0].TargetEntity, ShouldResemble, testEntityID)
			})
			Convey("and fetch back something that doesn't exist", func() {
				missingEntity := entity.New("test_post", newTestUserID()+"_missing")
				shares, err := cl.GetSharesByAuthor(ctx, testUserID, []entity.Entity{missingEntity})
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 0)
			})
			Convey("and fetch summary", func() {
				shareSummary, err := cl.GetSharesSummaries(ctx, []entity.Entity{testEntityID})
				So(err, ShouldBeNil)
				So(shareSummary.Items[0].ParentEntity, ShouldResemble, testEntityID)
				So(shareSummary.Items[0].Total, ShouldEqual, 1)
			})
			Convey("and fetch multiple back", func() {
				shares, err := cl.GetShares(ctx, []string{share.ID})
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 1)
				So(shares.Items[0].TargetEntity, ShouldResemble, testEntityID)
			})
			Convey("and fetch multiple missing should fail", func() {
				shares, err := cl.GetShares(ctx, []string{share.ID + "_MISSING_"})
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 0)
			})
			Convey("and delete it", func() {
				share2, err := cl.DeleteShare(ctx, share.ID)
				So(err, ShouldBeNil)
				So(share2.TargetEntity, ShouldResemble, testEntityID)
				Convey("and it should be gone", func() {
					share3, err := cl.DeleteShare(ctx, share.ID)
					So(err, ShouldBeNil)
					So(share3, ShouldBeNil)

					share3, err = cl.GetShare(ctx, share.ID)
					So(err, ShouldBeNil)
					So(share3, ShouldBeNil)
				})
				Convey("and fetch it back by entity should be nil", func() {
					shares, err := cl.GetSharesByAuthor(ctx, testUserID, []entity.Entity{testEntityID})
					So(err, ShouldBeNil)
					So(len(shares.Items), ShouldEqual, 0)
				})
				Convey("and fetch summary should have zero total shares", func() {
					shareSummary, err := cl.GetSharesSummaries(ctx, []entity.Entity{testEntityID})
					So(err, ShouldBeNil)
					So(shareSummary.Items[0].ParentEntity, ShouldResemble, testEntityID)
					So(shareSummary.Items[0].Total, ShouldEqual, 0)
				})
			})
		})
	})
}

func TestClientIntegration_Reaction(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		testUserID := newTestUserID()

		Convey("Should be able to create, get, and delete a reaction", func() {
			parent := entity.New("test_parent", timestamp())
			testEmoteID := "Kappa"
			created, err := cl.CreateReaction(ctx, parent, testUserID, testEmoteID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ParentEntity, ShouldResemble, parent)
			So(created.EmoteIDs, ShouldContain, testEmoteID)

			reaction, err := cl.GetReactions(ctx, parent, testUserID)
			So(err, ShouldBeNil)
			So(reaction, ShouldNotBeNil)

			deleted, err := cl.DeleteReaction(ctx, parent, testUserID, testEmoteID)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)
			So(len(deleted.EmoteIDs), ShouldEqual, 0)
		})
	})
}

func TestClientIntegration_Reaction_Summary(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		testUserID := newTestUserID()

		Convey("Should update the reaction summary when reactions are created and deleted", func() {
			parent := entity.New("test_parent", timestamp())

			cl.CreateReaction(ctx, parent, testUserID, "kappa")
			s1, err := cl.GetReactionsSummary(ctx, parent)
			So(err, ShouldBeNil)
			So(s1, ShouldNotBeNil)
			So(s1.Emotes, ShouldNotBeNil)
			So(s1.Emotes["kappa"], ShouldEqual, 1)

			cl.DeleteReaction(ctx, parent, testUserID, "kappa")
			s2, err := cl.GetReactionsSummary(ctx, parent)
			So(err, ShouldBeNil)
			So(s2, ShouldNotBeNil)
			So(len(s2.Emotes), ShouldEqual, 0)
		})

		Convey("Should get multiple reactions summaries by parent entity", func() {
			parent1 := entity.New("test_parent", timestamp())
			c1, err := cl.CreateReaction(ctx, parent1, testUserID, "120")
			So(err, ShouldBeNil)
			So(c1, ShouldNotBeNil)

			parent2 := entity.New("test_parent", timestamp())
			c2, err := cl.CreateReaction(ctx, parent2, testUserID, "121")
			So(err, ShouldBeNil)
			So(c2, ShouldNotBeNil)

			reactions, err := cl.GetReactionsSummariesByParent(ctx, []entity.Entity{parent1, parent2}, nil)
			So(err, ShouldBeNil)
			So(reactions, ShouldNotBeNil)
			So(len(reactions.Items), ShouldEqual, 2)
			Convey("and get by user id", func() {
				options := duplo.GetReactionsSummariesByParentOptions{
					UserIDs: []string{testUserID},
				}
				reactions, err := cl.GetReactionsSummariesByParent(ctx, []entity.Entity{parent1, parent2}, &options)
				So(err, ShouldBeNil)
				So(reactions, ShouldNotBeNil)
				So(len(reactions.Items), ShouldEqual, 2)
				for _, item := range reactions.Items {
					for _, sum := range item.EmoteSummaries {
						So(sum.UserIDs, ShouldResemble, []string{testUserID})
					}
				}
			})
		})

		Convey("Deleting non-existent reaction should return nil", func() {
			parent := entity.New("test:parent", timestamp())
			reactions, err := cl.DeleteReaction(ctx, parent, testUserID, "kappa")
			So(reactions, ShouldBeNil)
			So(err, ShouldBeNil)
		})
	})
}

type TestSetup struct {
	context.Context
	*duplo.Client
	cancelFunc func()
	Common     service_common.ConfigCommon
}

func (t *TestSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)
	if err := t.Common.Setup(); err != nil {
		return err
	}
	cfg := &duplo.Config{}
	if err := cfg.Load(t.Common.Config); err != nil {
		return err
	}
	t.Client = &duplo.Client{
		Config: cfg,
	}
	return nil
}

func (c *TestSetup) Close() error {
	c.cancelFunc()
	return c.Common.Close()
}
