// +build integration

package edge_test

import (
	"strconv"
	"testing"
	"time"

	"net/http"

	"code.justin.tv/feeds/clients/edge"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func newSetup() *TestSetup {
	return &TestSetup{
		Common: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "client_int_tests",
			BaseDirectory: "../",
		},
	}
}

const (
	testUserID     = "27184041"
	testChannelID  = testUserID
	testUserID2    = "181631"
	testChannelID2 = testUserID2

	// These are used to spread out tests because of post creation rate limiting
	// TODO We should use integration environments with low post creation timeouts
	// qa_arnold
	testUserID3 = "140837659"
	// rando
	testUserID4 = "46491334"
	// turtleeatspigeon4
	testUserID5 = "140601908"
	// turtleeatspigeon3
	testUserID6 = "140601818"
	// tubbatubtub - andrew's test account
	testUserID7 = "129660415"
	// w1fescape - andrew's other test account
	testUserID8    = "152973670"
	testEmoteKappa = "25"
)

// Config configures edge http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("edge.http_endpoint", "").Get
	return nil
}

func timestamp() string {
	return strconv.Itoa(int(time.Now().UnixNano()))
}

func TestClientIntegration_Share(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		t.Log("using client addr ", setup.Config.Host())

		Convey("After creating a post and share", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID

			created, err := cl.CreatePost(ctx, channelID, testBody, userID, nil, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)
			So(created.ShareSummary.ShareCount, ShouldEqual, 0)
			So(created.ShareSummary.UserIDs, ShouldBeEmpty)
			createdEntity := entity.New(entity.NamespacePost, created.ID)

			shareAuthor := newUserID()
			share, err := cl.CreateShare(ctx, shareAuthor, createdEntity, nil)
			So(err, ShouldBeNil)
			So(share.TargetEntity, ShouldResemble, createdEntity)

			Convey("should see share when getting post back", func() {
				secondPost, err := cl.GetPost(ctx, created.ID, nil, nil)
				So(err, ShouldBeNil)
				So(secondPost.ShareSummary.ShareCount, ShouldEqual, 1)
			})

			Convey("should be able to fetch it back", func() {
				share2, err := cl.GetShare(ctx, shareAuthor, share.ID, nil)
				So(err, ShouldBeNil)
				So(share2.UserID, ShouldEqual, shareAuthor)

				shareSummaries, err := cl.GetSharesSummaries(ctx, userID, []entity.Entity{createdEntity}, []string{shareAuthor}, nil)
				So(err, ShouldBeNil)
				So(shareSummaries.Items[0].ShareCount, ShouldEqual, 1)
				So(shareSummaries.Items[0].UserIDs, ShouldResemble, []string{shareAuthor})
			})
			Convey("should be able to delete it by target", func() {
				unsharedItems, err := cl.UnshareByTarget(ctx, shareAuthor, shareAuthor, []entity.Entity{createdEntity}, nil)
				So(err, ShouldBeNil)
				So(unsharedItems.Items[0].ID, ShouldEqual, share.ID)

				_, err = cl.GetShare(ctx, userID, share.ID, nil)
				So(errorCode(err), ShouldEqual, http.StatusNotFound)

				unsharedItems, err = cl.UnshareByTarget(ctx, shareAuthor, shareAuthor, []entity.Entity{createdEntity}, nil)
				So(err, ShouldBeNil)
				So(unsharedItems.Items, ShouldBeEmpty)
			})
			Convey("should be able to delete it", func() {
				share2, err := cl.DeleteShare(ctx, shareAuthor, share.ID, nil)
				So(err, ShouldBeNil)
				So(share2.UserID, ShouldEqual, shareAuthor)

				Convey("and it stay deleted", func() {
					_, err := cl.DeleteShare(ctx, shareAuthor, share.ID, nil)
					So(errorCode(err), ShouldEqual, http.StatusNotFound)

					_, err = cl.GetShare(ctx, shareAuthor, share.ID, nil)
					So(errorCode(err), ShouldEqual, http.StatusNotFound)

					shareSummaries, err := cl.GetSharesSummaries(ctx, userID, []entity.Entity{createdEntity}, []string{userID}, nil)
					So(err, ShouldBeNil)
					So(shareSummaries.Items[0].ShareCount, ShouldEqual, 0)
					So(shareSummaries.Items[0].UserIDs, ShouldBeEmpty)

					unsharedItems, err := cl.UnshareByTarget(ctx, shareAuthor, shareAuthor, []entity.Entity{createdEntity}, nil)
					So(err, ShouldBeNil)
					So(unsharedItems.Items, ShouldBeEmpty)

					Convey("should see share gone when getting post back", func() {
						secondPost, err := cl.GetPost(ctx, created.ID, nil, nil)
						So(err, ShouldBeNil)
						So(secondPost.ShareSummary.ShareCount, ShouldEqual, 0)
					})
				})
			})
		})
	})
}

func TestClientIntegration_Feed(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func(c C) {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		c.Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		cl := setup.Client
		t.Log("using client addr ", setup.Config.Host())

		Convey("Should be able to get an empty feed", func() {
			feedEntity := entity.New("c", timestamp())
			feed, err := cl.GetFeed(ctx, feedEntity, nil, nil)
			So(err, ShouldBeNil)
			So(feed, ShouldNotBeNil)
		})

		Convey("Should be able to create a post", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID

			created, err := cl.CreatePost(ctx, channelID, testBody, userID, nil, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)
		})

		Convey("Should be able to create a post with embedURLs", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID
			embedURLs := []string{"https://www.youtube.com/watch?v=dQw4w9WgXcQ"}

			createPostOptions := &edge.CreatePostOptions{
				EmbedURLs: &embedURLs,
			}

			created, err := cl.CreatePost(ctx, channelID, testBody, userID, createPostOptions, nil)
			So(err, ShouldBeNil)
			So(created.Embeds, ShouldHaveLength, 1)
			So(created.Embeds[0].RequestURL, ShouldEqual, embedURLs[0])
		})

		Convey("Should be able to create a syndicated post", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID

			created, err := cl.CreateSyndicatedPost(ctx, channelID, testBody, userID, false, nil, nil)
			So(err, ShouldBeNil)

			post := created.Post
			So(post.ID, ShouldNotBeNil)
			So(post.Body, ShouldEqual, testBody)

			So(created.Tweet, ShouldBeEmpty)
			So(created.TweetStatus, ShouldEqual, 0)
		})

		Convey("Should be able to create a syndicated post with embedURLs", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID
			embedURLs := []string{"https://www.youtube.com/watch?v=dQw4w9WgXcQ"}

			createPostOptions := &edge.CreatePostOptions{
				EmbedURLs: &embedURLs,
			}

			created, err := cl.CreateSyndicatedPost(ctx, channelID, testBody, userID, false, createPostOptions, nil)
			So(err, ShouldBeNil)
			So(created.Post.Embeds, ShouldHaveLength, 1)
			So(created.Post.Embeds[0].RequestURL, ShouldEqual, embedURLs[0])
		})

		Convey("Should be able to get permissions", func() {
			userID := "123456"
			permissions, err := cl.GetPostPermissions(ctx, userID, userID, nil)
			So(err, ShouldBeNil)
			So(permissions.CanShare, ShouldBeFalse)
		})

		Convey("Should be able to report a post", func() {
			// Create a post.
			testBody := "Testing reporting posts " + timestamp()
			userID := testUserID5
			channelID := userID

			created, err := cl.CreatePost(ctx, channelID, testBody, userID, nil, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)

			// Report the post for moderation.
			err = cl.ReportPost(ctx, created.ID, testUserID, "other", "1234556", nil)
			So(err, ShouldBeNil)
		})

		Convey("Should be able to get a post", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID

			created, err := cl.CreatePost(ctx, channelID, testBody, userID, nil, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)

			post, err := cl.GetPost(ctx, created.ID, nil, nil)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Body, ShouldEqual, testBody)
		})

		Convey("Should be able to delete a post", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			userID := newUserID()
			channelID := userID

			// Create a post.
			createdPost, err := cl.CreatePost(ctx, channelID, testBody, userID, nil, nil)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)

			// Delete the post.
			deletedPost, err := cl.DeletePost(ctx, createdPost.ID, userID, nil)
			So(err, ShouldBeNil)
			So(deletedPost, ShouldNotBeNil)
			So(deletedPost.Body, ShouldEqual, testBody)

			// Verify that the post is no longer accessible.
			getPostOptions := &edge.GetPostOptions{
				UserID: userID,
			}
			_, err = cl.GetPost(ctx, createdPost.ID, getPostOptions, nil)
			So(err, ShouldNotBeNil)
			So(errorCode(err), ShouldEqual, http.StatusNotFound)
		})

		Convey("Should be able to get embed data", func() {
			testURL := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			embed, err := cl.GetEmbed(ctx, testURL, nil, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.Type, ShouldEqual, "video")
		})

		Convey("Should be able to get set embedded video autoplay param to false", func() {
			testURL := "https://www.twitch.tv/qa_flv_vods_transcoded/v/43848179"
			autoplay := false
			options := &edge.GetEmbedOptions{Autoplay: &autoplay}
			embed, err := cl.GetEmbed(ctx, testURL, options, nil)
			So(err, ShouldBeNil)
			So(embed, ShouldNotBeNil)
			So(embed.PlayerHTML, ShouldContainSubstring, "autoplay=false")
		})

		Convey("Should be able to create a reaction", func() {
			userID := testUserID
			channelID := userID
			emoteID := testEmoteKappa

			// Create a post.
			createdPost, err := cl.CreatePost(ctx, channelID, "arbitrary post body", userID, nil, nil)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)
			postEntity := entity.New(entity.NamespacePost, createdPost.ID)

			// Create a reaction on the post.
			userReactions, err := cl.CreateReaction(ctx, postEntity, emoteID, userID, nil)

			So(err, ShouldBeNil)
			So(userReactions, ShouldNotBeNil)
			So(userReactions.EmoteIDs, ShouldContain, emoteID)
			So(userReactions.UserID, ShouldEqual, userID)
		})

		Convey("Should be able to get reactions from multiple parent entities", func() {
			userID := testUserID
			emoteID := testEmoteKappa

			// Create entities to which you can react.
			entity1 := entity.New("test1", timestamp())
			entity2 := entity.New("test1", timestamp())

			// Create a reaction on the first entity.
			_, err := cl.CreateReaction(ctx, entity1, emoteID, userID, nil)
			So(err, ShouldBeNil)

			// Create a reaction on the second entity.
			_, err = cl.CreateReaction(ctx, entity2, emoteID, userID, nil)
			So(err, ShouldBeNil)

			parentEntities := []entity.Entity{entity1, entity2}

			// Fetch reactions for both entity structs.
			fetchedReactions, err := cl.GetReactionsForParents(ctx, parentEntities, nil, nil)

			So(err, ShouldBeNil)
			So(fetchedReactions, ShouldNotBeNil)
			So(len(fetchedReactions.Reactions), ShouldEqual, 2)

			reactionMap := make(map[entity.Entity]map[string]*edge.ReactionToItem)
			for _, reaction := range fetchedReactions.Reactions {
				reactionMap[reaction.ParentEntity] = reaction.Reactions
			}

			So(reactionMap[entity1], ShouldContainKey, emoteID)
			So(reactionMap[entity2], ShouldContainKey, emoteID)
		})

		Convey("Should be able to get a user's reactions from multiple parent entities", func() {
			userID1 := testUserID
			userID2 := testUserID2
			emoteID := testEmoteKappa

			// Create entities to which you can react.
			entity1 := entity.New("test1", timestamp())
			entity2 := entity.New("test1", timestamp())

			// Create a reaction on the first entity as userID1.
			_, err := cl.CreateReaction(ctx, entity1, emoteID, userID1, nil)
			So(err, ShouldBeNil)

			// Create a reaction on the second entity as userID2.
			_, err = cl.CreateReaction(ctx, entity2, emoteID, userID2, nil)
			So(err, ShouldBeNil)

			parentEntities := []entity.Entity{entity1, entity2}

			// Fetch reactions for both entity structs as userID1.
			fetchedReactions, err := cl.GetReactionsForParents(ctx, parentEntities, &edge.GetReactionsForParentsOptions{UserID: userID1}, nil)

			So(err, ShouldBeNil)
			So(fetchedReactions, ShouldNotBeNil)
			So(len(fetchedReactions.Reactions), ShouldEqual, 2)

			reactionMap := make(map[entity.Entity]map[string]*edge.ReactionToItem)
			for _, reaction := range fetchedReactions.Reactions {
				reactionMap[reaction.ParentEntity] = reaction.Reactions
			}

			So(reactionMap[entity1], ShouldContainKey, emoteID)
			// The response should indicate that the caller (userID1) created this emote.
			So(reactionMap[entity1][emoteID].UserIDs, ShouldContain, userID1)

			So(reactionMap[entity2], ShouldContainKey, emoteID)
			// The response should indicate that the caller (userID1) did not create this emote.
			So(reactionMap[entity2][emoteID].UserIDs, ShouldNotContain, userID1)
		})

		Convey("Should be able to delete a reaction", func() {
			userID := testUserID
			channelID := userID
			emoteID := testEmoteKappa

			// Create a reaction on a post.
			createdPost, err := cl.CreatePost(ctx, channelID, "arbitrary post body", userID, nil, nil)
			So(err, ShouldBeNil)
			So(createdPost, ShouldNotBeNil)
			postEntity := entity.New(entity.NamespacePost, createdPost.ID)

			_, err = cl.CreateReaction(ctx, postEntity, emoteID, userID, nil)
			So(err, ShouldBeNil)

			// Delete the reaction.
			deletedReactions, err := cl.DeleteReaction(ctx, postEntity, emoteID, userID, nil)

			So(err, ShouldBeNil)
			So(deletedReactions, ShouldNotBeNil)
			So(deletedReactions.EmoteIDs, ShouldNotContain, emoteID)
			So(deletedReactions.UserID, ShouldEqual, userID)
		})

		Convey("Should be able to get settings", func() {
			settingsEntity := entity.New(entity.NamespaceUser, newUserID())

			settings, err := cl.GetSettings(ctx, settingsEntity, settingsEntity.ID(), nil)
			So(err, ShouldBeNil)
			So(settings, ShouldNotBeNil)
			So(settings.Entity, ShouldResemble, settingsEntity)
		})

		Convey("Should be able to update settings", func() {
			settingsEntity := entity.New(entity.NamespaceUser, newUserID())

			settings, err := cl.GetSettings(ctx, settingsEntity, settingsEntity.ID(), nil)
			So(err, ShouldBeNil)
			So(settings.Entity, ShouldResemble, settingsEntity)
			So(settings.AdminDisabledComments, ShouldBeFalse) // default value

			newAdminDisabledComments := true
			updated, err := cl.UpdateSettings(ctx, settingsEntity, settingsEntity.ID(), &edge.UpdateSettingsOptions{
				AdminDisabledComments: &newAdminDisabledComments,
			}, nil)
			So(err, ShouldBeNil)
			So(updated.Entity, ShouldResemble, settingsEntity)
			So(updated.AdminDisabledComments, ShouldEqual, newAdminDisabledComments)
		})
	})
}

type TestSetup struct {
	context.Context
	edge.Client
	cancelFunc func()
	Common     service_common.ConfigCommon
	Config     *Config
}

func (t *TestSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)
	if err := t.Common.Setup(); err != nil {
		return err
	}
	t.Config = &Config{}
	if err := t.Config.Load(t.Common.Config); err != nil {
		return err
	}

	var err error
	t.Client, err = edge.NewClient(twitchclient.ClientConf{
		Host: t.Config.Host(),
	})
	if err != nil {
		return err
	}

	return nil
}

func (c *TestSetup) Close() error {
	c.cancelFunc()
	return c.Common.Close()
}

func newUserID() string {
	return timestamp()
}

func errorCode(err error) int {
	if coded, ok := errors.Cause(err).(service_common.ErrorWithCode); ok {
		return coded.HTTPCode()
	}
	return 0
}
