package feedsedge

import (
	"io"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/foundation/twitchclient"
	"github.com/google/go-querystring/query"
	"golang.org/x/net/context"
)

const (
	defaultTimingXactName = "feeds-edge"
	defaultStatSampleRate = 1.0
)

// Client is the interface third parties should expect the edge client to implement
type Client interface {
	GetFeed(ctx context.Context, feed entity.Entity, options *GetFeedOptions, reqOpts *twitchclient.ReqOpts) (*Feed, error)

	GetPostsByIDs(ctx context.Context, postIDs []string, reqOpts *twitchclient.ReqOpts) (*Posts, error)
	GetPostsPermissionsByIDs(ctx context.Context, postIDs []string, userID string, reqOpts *twitchclient.ReqOpts) (*PostsPermissions, error)
	CreatePost(ctx context.Context, channelID string, body string, userID string, postToTwitter bool, EmbedURLs *[]string, reqOpts *twitchclient.ReqOpts) (*CreatePostResponse, error)
	DeletePost(ctx context.Context, postID string, userID string, reqOpts *twitchclient.ReqOpts) (*Post, error)

	GetSharesByIDs(ctx context.Context, shareIDs []string, reqOpts *twitchclient.ReqOpts) (*Shares, error)
	CreateShare(ctx context.Context, targetEntity entity.Entity, userID string, reqOpts *twitchclient.ReqOpts) (*Share, error)
	DeleteShare(ctx context.Context, shareID string, userID string, reqOpts *twitchclient.ReqOpts) (*Share, error)

	GetSettings(ctx context.Context, e entity.Entity, userID string, reqOpts *twitchclient.ReqOpts) (*Settings, error)
	UpdateSettings(ctx context.Context, e entity.Entity, userID string, options *UpdateSettingsOptions, reqOpts *twitchclient.ReqOpts) (*Settings, error)

	GetReactionsSummariesByEntities(ctx context.Context, entities []string, options *GetReactionsSummariesOptions, reqOpts *twitchclient.ReqOpts) (*ReactionsSummaries, error)
	CreateReaction(ctx context.Context, ent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) error
	DeleteReaction(ctx context.Context, ent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) error

	GetSuggestedFeeds(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*FeedIDs, error)
}

// ClientImpl implements the Client interface and uses the twitchclient client to make http requests
type ClientImpl struct {
	Client twitchclient.Client
}

var _ Client = &ClientImpl{}

// NewClient returns a new instance of the Client which uses the given config
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &ClientImpl{Client: twitchClient}, err
}

// Feed is a feed of entities
type Feed struct {
	Items []FeedItem `json:"items"`
}

// FeedItem is the trimmed down entity item for a feed
type FeedItem struct {
	Entity   entity.Entity     `json:"entity"`
	Reasons  []*FeedItemReason `json:"reasons,omitempty"`
	Tracking *FeedItemTracking `json:"tracking,omitempty"`
	Cursor   string            `json:"cursor"`
}

// FeedItemReason should be every reason that can happen.  Honestly, this is the easiest way to do this in Go
type FeedItemReason struct {
	Type   ReasonTypeEnum `json:"type"`
	UserID string         `json:"user_id,omitempty"`
}

// ReasonTypeEnum is the set of reason types we expose for feed items
type ReasonTypeEnum string

const (
	statsdPrefix = `service.feeds_edge.`

	// FollowedType indicates that a piece of content is in a user's feed because the user
	// followed the content creator.
	FollowedType ReasonTypeEnum = "followed"

	// ViewedType indicates that a piece of content is in a user's feed because the user
	// viewed similar content.
	ViewedType ReasonTypeEnum = "viewed"

	// PopularType indicates that a piece of content is in a user's feed because it is
	// popular on Twitch.
	PopularType ReasonTypeEnum = "popular"
)

// FeedItemTracking contains feed item level fields used for tracking
type FeedItemTracking struct {
	RecGenerationID    string `json:"rec_generation_id,omitempty"`
	RecGenerationIndex *int   `json:"rec_generation_index,omitempty"`
	CardImpressionID   string `json:"card_impression_id,omitempty"`
	BatchID            string `json:"batch_id,omitempty"`
}

// FeedIDs is a list of encoded feed IDs
type FeedIDs struct {
	FeedIDs []string `json:"feed_ids"`
}

// Posts is a set of Post items
type Posts struct {
	Items []*Post `json:"items"`
}

// Post is a post, but only contains what is returned from duplo
type Post struct {
	ID            string           `json:"id"`
	CreatedAt     time.Time        `json:"created_at"`
	Deleted       bool             `json:"deleted,omitempty"`
	UserID        string           `json:"user_id"`
	Body          string           `json:"body"`
	Emotes        []*Emote         `json:"emotes"`
	EmbedEntities *[]entity.Entity `json:"embed_entities,omitempty"`
}

// PostPermissions is the permissions for a single post.
type PostPermissions struct {
	PostID    string `json:"post_id"`
	CanDelete bool   `json:"can_delete"`
}

// PostsPermissions is the permissions for multiple posts.
type PostsPermissions struct {
	Items []PostPermissions `json:"items"`
}

// CreatePostResponse is Post with twitter response
type CreatePostResponse struct {
	Post        *Post  `json:"post,omitempty"`
	TweetStatus int    `json:"tweet_status"`
	Tweet       string `json:"tweet"`
}

// Emote is the emote object for the API
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

// Shares is a set of Share items
type Shares struct {
	Items []*Share `json:"items"`
}

// Share is the returned object for a database share
type Share struct {
	ID           string        `json:"id"`
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
	CreatedAt    time.Time     `json:"created_at"`
	DeletedAt    *time.Time    `json:"deleted_at,omitempty"`
}

// Settings contains user settings
type Settings struct {
	Entity                entity.Entity `json:"entity"` // User ID or feed ID
	CreatedAt             time.Time     `json:"created_at"`
	UpdatedAt             time.Time     `json:"updated_at"`
	SubsCanComment        bool          `json:"subs_can_comment"`
	FriendsCanComment     bool          `json:"friends_can_comment"`
	FollowersCanComment   bool          `json:"followers_can_comment"`
	UserDisabledComments  bool          `json:"user_disabled_comments"`
	AdminDisabledComments bool          `json:"admin_disabled_comments"`
	ChannelFeedEnabled    bool          `json:"channel_feed_enabled"`
}

// UpdateSettingsOptions specifies the optional parameters of an UpdateSettings operation
type UpdateSettingsOptions struct {
	SubsCanComment        *bool `json:"subs_can_comment,omitempty"`
	FriendsCanComment     *bool `json:"friends_can_comment,omitempty"`
	FollowersCanComment   *bool `json:"followers_can_comment,omitempty"`
	UserDisabledComments  *bool `json:"user_disabled_comments,omitempty"`
	AdminDisabledComments *bool `json:"admin_disabled_comments,omitempty"`
	ChannelFeedEnabled    *bool `json:"channel_feed_enabled,omitempty"`
}

// TwitchHTTPDoer does a Do(request) call using twitch HTTP client
type TwitchHTTPDoer struct {
	Client twitchclient.Client
	Reqopt twitchclient.ReqOpts
}

// Do a http request using the twitch client
func (t *TwitchHTTPDoer) Do(req *http.Request) (*http.Response, error) {
	return t.Client.Do(req.Context(), req, t.Reqopt)
}

// NewTwitchRequest implements NewHTTPRequest for twitchclient.Client
func (t *TwitchHTTPDoer) NewTwitchRequest(ctx context.Context, method string, url string, body io.Reader) (*http.Request, error) {
	req, err := t.Client.NewRequest(method, url, body)
	if err != nil {
		return nil, err
	}
	req = req.WithContext(ctx)
	return req, nil
}

func (c *ClientImpl) http(ctx context.Context, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchclient.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       statName,
		StatSampleRate: defaultStatSampleRate,
	})

	doer := &TwitchHTTPDoer{
		Client: c.Client,
		Reqopt: combinedReqOpts,
	}
	return clients.DoHTTP(ctx, doer, method, path, queryParams, body, into, doer.NewTwitchRequest)
}

// GetFeedOptions specifies the optional parameters of a v2 GetFeed operation
type GetFeedOptions struct {
	UserID   string `url:"user_id"`
	Limit    int    `url:"limit"`
	Cursor   string `url:"cursor"`
	DeviceID string `url:"device_id"`
	Language string `url:"language"`
}

// GetFeed returns the entities of a feed
func (c *ClientImpl) GetFeed(ctx context.Context, feed entity.Entity, options *GetFeedOptions, reqOpts *twitchclient.ReqOpts) (*Feed, error) {
	path := "/v2/get_feed"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}

	query.Add("feed_id", feed.Encode())

	var ret Feed
	if err := c.http(ctx, statsdPrefix+path, "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetPostsByIDs returns post information provided by duplo
func (c *ClientImpl) GetPostsByIDs(ctx context.Context, postIDs []string, reqOpts *twitchclient.ReqOpts) (*Posts, error) {
	path := "/v2/get_posts_by_ids"

	query := url.Values{}
	query.Set("post_ids", strings.Join(postIDs, ","))

	var ret Posts
	if err := c.http(ctx, statsdPrefix+path, "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetPostsPermissionsByIDs returns posts' permissions. The posts are given by IDs.
func (c *ClientImpl) GetPostsPermissionsByIDs(ctx context.Context, postIDs []string, userID string, reqOpts *twitchclient.ReqOpts) (*PostsPermissions, error) {
	path := "/v2/get_posts_permissions_by_ids"

	query := url.Values{}
	query.Set("user_id", userID)
	query.Set("post_ids", strings.Join(postIDs, ","))

	var ret PostsPermissions
	if err := c.http(ctx, statsdPrefix+path, "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

type createPostBody struct {
	UserID        string    `json:"user_id"`
	Body          string    `json:"body"`
	PostToTwitter bool      `json:"post_to_twitter"`
	EmbedURLs     *[]string `json:"embed_urls,omitempty"`
}

// CreatePost creates a post using the new API
func (c *ClientImpl) CreatePost(ctx context.Context, channelID string, body string, userID string, postToTwitter bool, embedURLs *[]string, reqOpts *twitchclient.ReqOpts) (*CreatePostResponse, error) {
	path := "/v2/create_post"

	query := url.Values{}
	query.Set("user_id", userID)

	httpBody := createPostBody{
		UserID:        channelID,
		Body:          body,
		PostToTwitter: postToTwitter,
		EmbedURLs:     embedURLs,
	}

	var ret CreatePostResponse
	err := c.http(ctx, statsdPrefix+path, "POST", path, query, httpBody, reqOpts, &ret)
	if err != nil {
		return nil, err
	}
	return &ret, nil
}

// DeletePost deletes a post using the new API
func (c *ClientImpl) DeletePost(ctx context.Context, postID string, userID string, reqOpts *twitchclient.ReqOpts) (*Post, error) {
	path := "/v2/delete_post"

	query := url.Values{}
	query.Set("post_id", postID)
	query.Set("user_id", userID)

	var ret Post
	err := c.http(ctx, statsdPrefix+path, "DELETE", path, query, nil, reqOpts, &ret)
	if err != nil {
		return nil, err
	}
	return &ret, nil
}

// GetSharesByIDs returns share information provided by duplo
func (c *ClientImpl) GetSharesByIDs(ctx context.Context, shareIDs []string, reqOpts *twitchclient.ReqOpts) (*Shares, error) {
	path := "/v2/get_shares_by_ids"

	query := url.Values{}
	query.Set("share_ids", strings.Join(shareIDs, ","))

	var ret Shares
	if err := c.http(ctx, statsdPrefix+path, "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

type createShareBody struct {
	TargetID string `json:"target_entity"`
}

// CreateShare creates a share
func (c *ClientImpl) CreateShare(ctx context.Context, targetEntity entity.Entity, userID string, reqOpts *twitchclient.ReqOpts) (*Share, error) {
	path := "/v2/create_share"

	query := url.Values{}
	query.Set("user_id", userID)

	httpBody := createShareBody{
		TargetID: targetEntity.Encode(),
	}

	var ret Share
	err := c.http(ctx, statsdPrefix+path, "POST", path, query, httpBody, reqOpts, &ret)
	if err != nil {
		return nil, err
	}
	return &ret, nil
}

// DeleteShare deletes a share
func (c *ClientImpl) DeleteShare(ctx context.Context, shareID string, userID string, reqOpts *twitchclient.ReqOpts) (*Share, error) {
	path := "/v2/delete_share"

	query := url.Values{}
	query.Set("share_id", shareID)
	query.Set("user_id", userID)

	var ret Share
	err := c.http(ctx, statsdPrefix+path, "DELETE", path, query, nil, reqOpts, &ret)
	if err != nil {
		return nil, err
	}
	return &ret, nil

}

// ReactionsSummaries is the response to a bulk get reactions
type ReactionsSummaries struct {
	Items []*ReactionSummaries `json:"items"`
}

// ReactionSummaries is the collection of all reaction summaries for the given parent entity
type ReactionSummaries struct {
	ParentEntity entity.Entity      `json:"parent_entity"`
	Summaries    []*ReactionSummary `json:"summaries"`
}

// ReactionSummary is the count for a particular emote on a given parent entity
type ReactionSummary struct {
	EmoteID     string `json:"emote_id"`
	EmoteName   string `json:"emote_name"`
	Count       int    `json:"count"`
	UserReacted bool   `json:"user_reacted"`
}

// GetReactionsSummariesOptions specifies the optional parameters of a v2 GetReactionsSummariesByEntities operation
type GetReactionsSummariesOptions struct {
	UserID string `url:"user_id"`
}

// GetReactionsSummariesByEntities returns reaction information provided by duplo
func (c *ClientImpl) GetReactionsSummariesByEntities(ctx context.Context, entities []string, options *GetReactionsSummariesOptions, reqOpts *twitchclient.ReqOpts) (*ReactionsSummaries, error) {
	path := "/v2/get_reactions_by_entities"

	vq, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	vq.Set("entities", strings.Join(entities, ","))

	var ret ReactionsSummaries
	if err := c.http(ctx, statsdPrefix+path, "GET", path, vq, nil, reqOpts, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetSettings gets settings for the given user
func (c *ClientImpl) GetSettings(ctx context.Context, e entity.Entity, userID string, reqOpts *twitchclient.ReqOpts) (*Settings, error) {
	path := "/v1/settings/" + e.Encode()

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Settings
	if err := c.http(ctx, statsdPrefix+"get/v1/settings/", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// GetSuggestedFeeds returns which feeds of feeds we think a user would be interested in
func (c *ClientImpl) GetSuggestedFeeds(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*FeedIDs, error) {
	path := "/v2/suggested_feeds"

	query := url.Values{}
	query.Set("user_id", userID)

	var ret FeedIDs
	if err := c.http(ctx, statsdPrefix+"get/v2/suggested_feeds/", "GET", path, query, nil, reqOpts, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// UpdateSettings updates settings for the given user
func (c *ClientImpl) UpdateSettings(ctx context.Context, e entity.Entity, userID string, options *UpdateSettingsOptions, reqOpts *twitchclient.ReqOpts) (*Settings, error) {
	path := "/v1/settings/" + e.Encode()

	query := url.Values{}
	query.Set("user_id", userID)

	var ret Settings
	if err := c.http(ctx, statsdPrefix+"post/v1/settings/", "POST", path, query, options, reqOpts, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// CreateReaction adds a reaction to an entity
func (c *ClientImpl) CreateReaction(ctx context.Context, ent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := "/v2/create_reaction"

	query := url.Values{}
	query.Set("user_id", userID)
	query.Set("parent_entity", ent.Encode())
	query.Set("emote_id", emoteID)

	var ret string
	return c.http(ctx, statsdPrefix+"put/v2/create_reaction/", "PUT", path, query, nil, reqOpts, &ret)
}

// DeleteReaction removes a reaction from an entity
func (c *ClientImpl) DeleteReaction(ctx context.Context, ent entity.Entity, emoteID, userID string, reqOpts *twitchclient.ReqOpts) error {
	path := "/v2/delete_reaction"

	query := url.Values{}
	query.Set("user_id", userID)
	query.Set("parent_entity", ent.Encode())
	query.Set("emote_id", emoteID)

	var ret string
	return c.http(ctx, statsdPrefix+"delete/v2/delete_reaction/", "DELETE", path, query, nil, reqOpts, &ret)
}
