package masonry

import (
	"bytes"
	"encoding/json"
	"fmt"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/ctxlog/ctxlogaws"
	"code.justin.tv/feeds/ctxlog/ctxlogaws/ctxlogsqs"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/net/context"
)

// QueueConfig configures masonry SQS client for normal priority messages
type QueueConfig struct {
	QueueURL func() string
}

func (c *QueueConfig) queueURL() string {
	if c == nil {
		return ""
	}
	return c.QueueURL()
}

func (c *QueueConfig) loadForQueue(dconf *distconf.Distconf, qLevel string) error {
	c.QueueURL = dconf.Str(fmt.Sprintf("masonry.%s.sqssource.queue_url", qLevel), "").Get
	if c.queueURL() == "" {
		return errors.Errorf("variable masonry.%s.sqssource.queue_url must be set", qLevel)
	}
	return nil
}

// LowPriorityQueueConfig configures masonry SQS client for low priority messages
type LowPriorityQueueConfig struct {
	QueueConfig
}

// Load configuration information
func (c *LowPriorityQueueConfig) Load(dconf *distconf.Distconf) error {
	return c.QueueConfig.loadForQueue(dconf, "low")
}

// MidPriorityQueueConfig configures masonry SQS client for mid priority messages
type MidPriorityQueueConfig struct {
	QueueConfig
}

// Load configuration information
func (c *MidPriorityQueueConfig) Load(dconf *distconf.Distconf) error {
	return c.QueueConfig.loadForQueue(dconf, "mid")
}

// HighPriorityQueueConfig configures masonry SQS client for high priority messages
type HighPriorityQueueConfig struct {
	QueueConfig
}

// Load configuration information
func (c *HighPriorityQueueConfig) Load(dconf *distconf.Distconf) error {
	return c.QueueConfig.loadForQueue(dconf, "high")
}

// QueueClient adds masonry feed items for later storage
type QueueClient struct {
	Sqs    *sqs.SQS
	Config *QueueConfig
	Ch     *ctxlog.Ctxlog
	Log    *log.ElevatedLog
}

// ActivityBatch gives masonry multiple activity actions for a single consumer
type ActivityBatch struct {
	Activities []*Activity `json:"activities"`
	FeedIDs    []string    `json:"feed_ids"`
	// An aggregate metadata for all the stories in the batch
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`
}

// QueueActivityBatch gives masonry a batch of activities and feed IDs to process.  It will attempt to add
// every activity given to every feed given.
func (c *QueueClient) QueueActivityBatch(ctx context.Context, batch *ActivityBatch) error {
	buf := &bytes.Buffer{}
	if err := json.NewEncoder(buf).Encode(batch); err != nil {
		return err
	}
	msgJSON := buf.String()
	input := &sqs.SendMessageInput{
		QueueUrl:          aws.String(c.Config.queueURL()),
		MessageBody:       &msgJSON,
		MessageAttributes: ctxlogsqs.ModifyRequest(ctx, nil, c.Ch),
	}
	req, _ := c.Sqs.SendMessageRequest(input)
	req.HTTPRequest = req.HTTPRequest.WithContext(ctx)
	return ctxlogaws.DoAWSSend(req, c.Log)
}
