package sfxconfig

import (
	"os"
	"regexp"

	"code.justin.tv/common/config"
	"code.justin.tv/feeds/metrics/multistatsd"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"github.com/cactus/go-statsd-client/statsd"
)

var invalidStatRegexp = regexp.MustCompile(`[^A-Za-z0-9-]+`)

func sanitizedHostname() string {
	hostname, err := os.Hostname()
	if err != nil {
		return "unknown"
	}
	return invalidStatRegexp.ReplaceAllString(hostname, "_")
}

// Setup a sfxstatsd result to take the place of config's default statsd with a multi sender that will send
// to both the default config and SignalFx.
//
// You *must* call this after calling config.Parse()
func Setup(setupResult *sfxstatsd.SetupResult) {
	defaultDims := make(map[string]string, 3)
	if config.App() != "" {
		defaultDims["service"] = config.App()
	}
	if config.Environment() != "" {
		defaultDims["environment"] = config.Environment()
	}

	if h := sanitizedHostname(); h != "" {
		defaultDims["hostname"] = h
	}

	setupResult.Scheduler.DefaultDimensions(defaultDims)

	if _, alreadySetup := config.Statsd().(*sfxstatsd.Statter); alreadySetup {
		return
	}

	config.SetStatsd(&multistatsd.MultiStatter{
		MultiStatSender: multistatsd.MultiStatSender{
			StatSenders: []statsd.StatSender{
				config.Statsd(), &setupResult.Statter,
			},
		},
	})
}
