package gomemcache

import (
	"context"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedcache"
	"code.justin.tv/foundation/gomemcache/memcache"
)

type GomemcachePool struct {
	Client *memcache.Client
}

type GomemcacheConfig struct {
	CacheTTL *distconf.Duration
}

func (g *GomemcacheConfig) Verify(prefix string, dconf *distconf.Distconf) error {
	g.CacheTTL = dconf.Duration(prefix+".ttl", time.Minute*15)
	return nil
}

func (g *GomemcacheConfig) keyTTL(s string) time.Duration {
	return g.CacheTTL.Get()
}

func NewGomemcache(client *memcache.Client, conf *GomemcacheConfig, keyPrefix string, Log *log.ElevatedLog, Stats *service_common.StatSender) *feedcache.ObjectCache {
	return &feedcache.ObjectCache{
		ClientPool: &GomemcachePool{
			Client: client,
		},
		Stats:     Stats,
		KeyPrefix: keyPrefix,
		KeyTTL:    conf.keyTTL,
		Log:       Log,
	}
}

var _ feedcache.CacheClientPool = &GomemcachePool{}

func (p *GomemcachePool) GetClient(ctx context.Context) feedcache.CacheClient {
	return &gomemcacheClient{Client: p.Client}
}

type gomemcacheClient struct {
	Client *memcache.Client
}

func (g *gomemcacheClient) Close(ctx context.Context) error {
	g.Client.Close()
	return nil
}

func (g *gomemcacheClient) Delete(ctx context.Context, key string) error {
	return g.Client.Delete(ctx, key)
}

func (g *gomemcacheClient) Get(ctx context.Context, key string) ([]byte, error) {
	item, err := g.Client.Get(ctx, key)
	if err != nil {
		return nil, err
	}
	// This is a bit of defensive coding.  Right now the client doesn't return nil items, but might as well check
	if item == nil {
		return nil, nil
	}
	return item.Value, nil
}

func (g *gomemcacheClient) Set(ctx context.Context, key string, value []byte, ttl time.Duration) error {
	return g.Client.Set(ctx, &memcache.Item{
		Key:        key,
		Value:      value,
		Expiration: int32(ttl.Seconds()),
	})
}
