package gorediscache

import (
	"time"

	"context"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedcache"
	"github.com/go-redis/redis"
)

type GoRedisPool struct {
	Client redis.Cmdable
}

type GoredisConfig struct {
	CacheTTL *distconf.Duration
}

func (g *GoredisConfig) Verify(prefix string, dconf *distconf.Distconf) error {
	g.CacheTTL = dconf.Duration(prefix+".ttl", time.Minute*15)
	return nil
}

func (g *GoredisConfig) keyTTL(s string) time.Duration {
	return g.CacheTTL.Get()
}

func NewGoredisCache(client redis.Cmdable, conf *GoredisConfig, keyPrefix string, Log *log.ElevatedLog, Stats *service_common.StatSender) *feedcache.ObjectCache {
	return &feedcache.ObjectCache{
		ClientPool: &GoRedisPool{
			Client: client,
		},
		Stats:     Stats,
		KeyPrefix: keyPrefix,
		KeyTTL:    conf.keyTTL,
		Log:       Log,
	}
}

var _ feedcache.CacheClientPool = &GoRedisPool{}

func (g *GoRedisPool) GetClient(ctx context.Context) feedcache.CacheClient {
	return &goRedisClient{Client: g.Client}
}

type goRedisClient struct {
	Client redis.Cmdable
}

func (g *goRedisClient) Close(ctx context.Context) error {
	return nil
}

func (g *goRedisClient) Delete(ctx context.Context, key string) error {
	return g.Client.Del(key).Err()
}

func (g *goRedisClient) Get(ctx context.Context, key string) ([]byte, error) {
	return g.Client.Get(key).Bytes()
}

func (g *goRedisClient) Set(ctx context.Context, key string, value []byte, ttl time.Duration) error {
	return g.Client.Set(key, value, ttl).Err()
}
