// +build integration

package feedcache

import (
	"context"
	"testing"
	"time"

	"code.justin.tv/feeds/errors"
	. "github.com/smartystreets/goconvey/convey"
)

func timestamp() string {
	return time.Now().Format(time.RFC3339Nano)
}

func CommonCacheTestingHelper(ctx context.Context, t *testing.T, cache *ObjectCache) {
	Convey("With a cache", func() {
		Convey("Should be able to cache an object", func() {
			key := "testCacheSuite_" + timestamp()

			type toCache struct {
				Name string
			}
			var thing toCache
			err := cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "cache_object"}, nil
			}, &thing)
			So(err, ShouldBeNil)
			So(thing.Name, ShouldEqual, "cache_object")

			var thing2 toCache
			err = cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "cache_object_not_seen"}, nil
			}, &thing2)
			So(err, ShouldBeNil)
			So(thing.Name, ShouldEqual, "cache_object")
		})

		Convey("Should be able to invalidate the cache", func() {
			key := "testCacheSuite_" + timestamp()

			type toCache struct {
				Name string
			}
			var thing toCache
			err := cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "invalidate_cache"}, nil
			}, &thing)
			So(err, ShouldBeNil)
			So(thing.Name, ShouldEqual, "invalidate_cache")

			err = cache.Invalidate(ctx, key)
			So(err, ShouldBeNil)

			var thing2 toCache
			err = cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "invalidate_cache_new"}, nil
			}, &thing2)
			So(err, ShouldBeNil)
			So(thing2.Name, ShouldEqual, "invalidate_cache_new")
		})

		Convey("Should be able to force cache something", func() {
			key := "testCacheSuite_" + timestamp()

			type toCache struct {
				Name string
			}
			var thing toCache
			err := cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "force_cache"}, nil
			}, &thing)
			So(err, ShouldBeNil)
			So(thing.Name, ShouldEqual, "force_cache")

			err = cache.ForceCached(ctx, key, toCache{Name: "force_cache_new"})
			So(err, ShouldBeNil)

			var thing2 toCache
			err = cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "force_cache_not_seen"}, nil
			}, &thing2)
			So(err, ShouldBeNil)
			So(thing2.Name, ShouldEqual, "force_cache_new")
		})

		Convey("Should not cache error values", func() {
			key := "testCacheSuite_" + timestamp()

			type toCache struct {
				Name string
			}
			var thing toCache
			cacheError := errors.New("error")
			err := cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "error_cache"}, cacheError
			}, &thing)
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldEqual, cacheError.Error())
			So(thing.Name, ShouldBeEmpty)

			var thing2 toCache
			err = cache.Cached(ctx, key, func() (interface{}, error) {
				return toCache{Name: "error_cache_new"}, nil
			}, &thing2)
			So(err, ShouldBeNil)
			So(thing2.Name, ShouldEqual, "error_cache_new")
		})

		Convey("Should be able to cache ints", func() {
			key := "testCacheSuite_" + timestamp()

			callbackCount := 0

			ints1, err := cache.Ints(ctx, key, func() ([]int64, error) {
				callbackCount++
				return []int64{1, 3}, nil
			})
			So(err, ShouldBeNil)
			So(ints1, ShouldResemble, []int64{1, 3})
			So(callbackCount, ShouldEqual, 1)

			ints2, err := cache.Ints(ctx, key, func() ([]int64, error) {
				callbackCount++
				return []int64{1, 5}, nil
			})
			So(err, ShouldBeNil)
			So(ints2, ShouldResemble, []int64{1, 3})
			So(callbackCount, ShouldEqual, 1)
		})
	})
}
