package feedsdynamo

import (
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"golang.org/x/net/context"
)

// KeyLookupFunc should return a searchable dynamodb row for a single unique string key
type KeyLookupFunc func(string) map[string]*dynamodb.AttributeValue

// KeyedBatchGet helps with Dynamo's KeyedBatchGet
// BatchGetItemInput has a few gotchas
//   1) You cannot pass duplicates
//   2) You cannot pass empty arrays
//   3) You cannot pass more than 100 items
//   4) A response may have unprocessed keys
//
//  People tend to forget all these, so this does them all at once.  It assumes a unique []string is the primary constraint
// on the batch get.
type KeyedBatchGet struct {
	// Dynamo is the interface to dynamodb
	Dynamo *dynamodb.DynamoDB
	// RequireConsistentReads if true, will require consistent reads per request
	RequireConsistentReads bool
	// ElevatedLog allows debug logging the request
	ElevatedLog *log.ElevatedLog
}

// BatchGet uses dynamodb to batch get a bunch of items
func (k *KeyedBatchGet) BatchGet(ctx context.Context, ids []string, tableName string, keyCreator KeyLookupFunc) ([]map[string]*dynamodb.AttributeValue, error) {
	ids = uniqueStrings(ids)
	if len(ids) == 0 {
		return nil, nil
	}

	// Can only call BatchGetItemInput on 100 items at once
	if len(ids) > 100 {
		firstResult, err := k.BatchGet(ctx, ids[:100], tableName, keyCreator)
		if err != nil {
			return nil, err
		}
		secondResult, err := k.BatchGet(ctx, ids[100:], tableName, keyCreator)
		if err != nil {
			return nil, err
		}
		return append(firstResult, secondResult...), nil
	}

	return k.loopKeys(ctx, ids, tableName, keyCreator)
}

func (k *KeyedBatchGet) setBatchGetItemConsistency(in *dynamodb.BatchGetItemInput) {
	if k.RequireConsistentReads {
		for _, v := range in.RequestItems {
			v.ConsistentRead = aws.Bool(true)
		}
	}
}

func uniqueStrings(ids []string) []string {
	ret := make([]string, 0, len(ids))
	requireUniqueKeys := make(map[string]struct{}, len(ids))
	for i := 0; i < len(ids); i++ {
		if _, exists := requireUniqueKeys[ids[i]]; exists {
			continue
		}
		requireUniqueKeys[ids[i]] = struct{}{}
		ret = append(ret, ids[i])
	}
	return ret
}

// loopKeys is a helper and should not be called directly.  It should only be called by keyedBatchGet
func (k *KeyedBatchGet) loopKeys(ctx context.Context, ids []string, tableName string, keyCreator KeyLookupFunc) ([]map[string]*dynamodb.AttributeValue, error) {
	keys := make([]map[string]*dynamodb.AttributeValue, 0, len(ids))
	for i := 0; i < len(ids); i++ {
		keys = append(keys, keyCreator(ids[i]))
	}
	ret := make([]map[string]*dynamodb.AttributeValue, 0, len(keys))
	unprocessedKeys := map[string]*dynamodb.KeysAndAttributes{
		tableName: {
			Keys: keys,
		},
	}

	// We keep looping as long as there are keys in the table we're interested in left to process
	for len(unprocessedKeys) > 0 {
		params := &dynamodb.BatchGetItemInput{
			RequestItems: unprocessedKeys,
		}
		k.setBatchGetItemConsistency(params)

		req, items := k.Dynamo.BatchGetItemRequest(params)

		if err := service_common.ContextSend(ctx, req, k.ElevatedLog); err != nil {
			return nil, err
		}
		if items.Responses == nil {
			return nil, nil
		}
		ret = append(ret, items.Responses[tableName]...)
		if len(items.UnprocessedKeys) > 0 && items.UnprocessedKeys[tableName] != nil && len(unprocessedKeys[tableName].Keys) <= len(items.UnprocessedKeys[tableName].Keys) {
			// We got back more items as unprocessed we requested???? This is very strange.
			return nil, errors.Errorf("appear to be in a batchitemget Ifiniloop: %d <= %d", len(unprocessedKeys[tableName].Keys), len(items.UnprocessedKeys[tableName].Keys))
		}
		unprocessedKeys = items.UnprocessedKeys
	}
	return ret, nil
}
