package service_common

import (
	"expvar"
	"fmt"
	"io/ioutil"
	"net"
	"net/http"
	"net/http/pprof"
	"os"
	"strings"
	"sync"
	"sync/atomic"
	"time"

	"encoding/json"

	"code.justin.tv/feeds/awsexpvar"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/expvar2"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/log/fmtlogger"
	"code.justin.tv/feeds/metrics/multistatsd"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdtruth"
	"code.justin.tv/feeds/rollbar"
	"code.justin.tv/feeds/rolllog"
	"code.justin.tv/feeds/xray"
	"code.justin.tv/feeds/xray/plugins/ec2"
	"code.justin.tv/hygienic/multistatsd/multistatsender"
	"code.justin.tv/hygienic/sandyconf"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/cep21/circuit"
	"github.com/cep21/circuit/closers/hystrix"
	"github.com/cep21/circuit/metriceventstream"
	"github.com/cep21/circuit/metrics/responsetimeslo"
	"github.com/cep21/circuit/metrics/rolling"
	"github.com/cep21/circuit/metrics/statsdmetrics"
	"github.com/signalfx/golib/datapoint"
	"github.com/signalfx/golib/sfxclient"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"
)

var SetupLogger = log.ElevatedLog{NormalLog: log.ContextLogger{Logger: fmtlogger.NewLogfmtLogger(os.Stdout, log.Discard)}}

// ServiceCommon is the root structure for all feeds services and sets up commonly needed things.
type ServiceCommon struct {
	ConfigCommon

	hasFinished bool
	stopChannel chan struct{}
	wg          sync.WaitGroup

	Secrets        *distconf.Distconf
	Log            *log.ElevatedLog
	Statsd         statsd.Statter         `nilcheck:"nodepth"`
	SfxSetupConfig sfxstatsd.SetupConfig  `nilcheck:"ignore"`
	SfxResult      *sfxstatsd.SetupResult `nilcheck:"ignore"`
	ErrorTracker   ErrorTracker
	ExpvarHandler  expvar2.Handler
	CodeVersion    string
	Stats          Stats
	StatsdTruth    statsdtruth.StatsdTruth `nilcheck:"ignore"`
	PanicLogger    PanicLogger
	XRay           *xray.XRay `nilcheck:"nodepth"`

	Circuit            circuit.Manager
	SLOConfig          map[string]responsetimeslo.Config
	CloserConfig       map[string]hystrix.ConfigureCloser
	OpenerConfig       map[string]hystrix.ConfigureOpener
	CircuitConfig      map[string]circuit.Config
	circuitEventStream *metriceventstream.MetricEventStream `nilcheck:"ignore"`

	gometrics     Gometrics       `nilcheck:"ignore"`
	rollbarClient *rollbar.Client `nilcheck:"ignore"`

	debugListener net.Listener

	setupBackgroundProcsOnce sync.Once
	statsPrefix              string
	setupSync                sync.RWMutex
	lastSetupErr             error
}

// Stats keep track of common stats during normal service common operation
type Stats struct {
	RollbarErrors int64
}

// CreateDefaultMux creates a goji mux that has /debug/running and a logging middleware
func CreateDefaultMux(Log *log.ElevatedLog, elevateKey interface{}, Ctxlog *ctxlog.Ctxlog, PanicLogger PanicLogger, dims *log.CtxDimensions, xRay *xray.XRay) *goji.Mux {
	mux := goji.NewMux()
	return PopulateMux(mux, Log, elevateKey, Ctxlog, PanicLogger, dims, xRay)
}

// PopulateMux populates a default mux that works for most twitch HTTP applications
func PopulateMux(mux *goji.Mux, Log *log.ElevatedLog, elevateKey interface{}, Ctxlog *ctxlog.Ctxlog, PanicLogger PanicLogger, dims *log.CtxDimensions, xRay *xray.XRay) *goji.Mux {
	logMiddleware := HTTPLoggerMiddleware{
		Log:         Log,
		PanicLogger: PanicLogger,
		CtxDim:      dims,
	}
	mux.Handle(pat.Get("/debug/running"), &HTTPStatusOk{
		Log: Log.NormalLog,
	})
	mux.Use(xRay.Handler)
	// Must be after xray handler: response with the trace ID
	mux.Use(xray.ResponseWithTraceID)
	mux.Use(func(h http.Handler) http.Handler {
		return &ctxlog.CtxHandler{
			Next:   h,
			Ctxlog: Ctxlog,
			Logger: Log,
			// Must be after the xray handler
			IDFetcher: xray.TraceID,
		}
	})
	mux.Use(logMiddleware.NewHandler)
	return mux
}

// CreateDefaultMux creates a default mux that works for most twitch HTTP applications
func (s *ServiceCommon) CreateDefaultMux() *goji.Mux {
	return CreateDefaultMux(s.Log, s.ElevateLogKey, &s.Ctxlog, s.PanicLogger, &s.CtxDimensions, s.XRay)
}

// PopulateMux populates a default mux that works for most twitch HTTP applications
func (s *ServiceCommon) PopulateMux(mux *goji.Mux) *goji.Mux {
	return PopulateMux(mux, s.Log, s.ElevateLogKey, &s.Ctxlog, s.PanicLogger, &s.CtxDimensions, s.XRay)
}

func (s *ServiceCommon) Format(f fmt.State, c rune) {
	fmt.Fprintf(f, "Service common: (%s)", s.Environment)
}

// Setup returns non nil if the service can setup itself correctly
func (s *ServiceCommon) Setup() (retErr error) {
	s.setupSync.Lock()
	defer s.setupSync.Unlock()
	if s.hasFinished {
		return s.lastSetupErr
	}
	defer func() {
		s.hasFinished = true
		s.lastSetupErr = retErr
	}()
	// For log messages that happen during setup, before a real logger is set
	if s.Log == nil {
		s.Log = &SetupLogger
		defer func() {
			if s.Log == &SetupLogger {
				s.Log = nil
			}
		}()
	}
	s.stopChannel = make(chan struct{})
	setups := []func() error{
		s.ConfigCommon.Setup,
		s.setupSecrets,
		s.setupLogging,
		s.setupStatsd,
		s.setupSignalfx,
		s.setupGometrics,
		s.setupXRay,
		s.setupHystrix,
		s.setupStatsdTruther,
		s.setupExvarHandler,
		s.setupDebugPort,
	}
	if s.CodeVersion == "" {
		s.CodeVersion = os.Getenv("GIT_COMMIT")
	}
	for _, setupFunc := range setups {
		err := setupFunc()
		if err != nil {
			return err
		}
	}
	s.Log.Log("code_version", s.CodeVersion, "pid", os.Getpid(), "Common setup completed")
	if s.Secrets == nil {
		s.Log.Log("No sandstorm setup.  Could not find correct ARN")
	}
	return nil
}

// SetupAdditionalSandstormDistconf returns a Distconf pointer that allows reading Sandstorm secrets from an alternative
// team/service bucket.  It is intended to facilitate setting up clients that need Sandstorm secrets managed by other
// teams.
func (s *ServiceCommon) SetupAdditionalSandstormDistconf(team, service string) *distconf.Distconf {
	readers := make([]distconf.Reader, 0, 1)
	if sandstormConf := s.registerSandstormConf(team, service); sandstormConf != nil {
		readers = append(readers, sandstormConf)
	}

	dc := &distconf.Distconf{
		Readers: readers,
		Logger: func(key string, err error, msg string) {
			s.Log.Log("key", key, "err", err, msg)
		},
	}

	// Register this distconf object with Expvar.  This allows us to view the keys that were loaded from this
	// sandstorm bucket.
	expvarKey := fmt.Sprintf("sandstorm_%s_%s", team, service)
	s.ExpvarHandler.Exported[expvarKey] = expvar.Func(func() interface{} {
		return dc.Keys()
	})

	return dc
}

// Close ends any listening ports and goroutines the service created
func (s *ServiceCommon) Close() error {
	close(s.stopChannel)
	s.setupBackgroundProcsOnce.Do(s.setupBackgroundProcs)
	errs := make([]error, 0, 4)
	if s.debugListener != nil {
		errs = append(errs, s.debugListener.Close())
	}
	if s.Statsd != nil {
		errs = append(errs, s.Statsd.Close())
	}
	errs = append(errs, s.StatsdTruth.Close())
	errs = append(errs, s.ConfigCommon.Close())
	if s.circuitEventStream != nil {
		errs = append(errs, s.circuitEventStream.Close())
	}

	if s.Secrets != nil {
		s.Secrets.Close()
	}
	s.wg.Wait()
	return ConsolidateErrors(errs)
}

func (s *ServiceCommon) setupStatsdTruther() error {
	s.StatsdTruth.SendTo = s.Statsd.NewSubStatter("truth")
	return nil
}

func (s *ServiceCommon) responseSLO(circuitName string) responsetimeslo.Config {
	if ret, exists := s.SLOConfig[circuitName]; exists {
		return ret
	}
	return responsetimeslo.Config{
		MaximumHealthyTime: s.Config.Duration("circuit."+circuitName+".slo.max_healthy", time.Millisecond*300).Get(),
	}
}

func (s *ServiceCommon) hystrixOpener(circuitName string) hystrix.ConfigureOpener {
	if ret, exists := s.OpenerConfig[circuitName]; exists {
		return ret
	}
	return hystrix.ConfigureOpener{
		ErrorThresholdPercentage: s.Config.Int("circuit."+circuitName+".opener.errorThresholdPercentage", 50).Get(),
		RequestVolumeThreshold:   s.Config.Int("circuit."+circuitName+".opener.requestVolumeThreshold", 20).Get(),
		RollingDuration:          s.Config.Duration("circuit."+circuitName+".opener.rollingDuration", time.Second*10).Get(),
		NumBuckets:               int(s.Config.Int("circuit."+circuitName+".opener.numBuckets", 10).Get()),
	}
}

func (s *ServiceCommon) circuitLostErrors(err error, panics interface{}) {
	s.Log.Log("err", err, "panic_result", panics)
}

func (s *ServiceCommon) circuitConfig(circuitName string) circuit.Config {
	if ret, exists := s.CircuitConfig[circuitName]; exists {
		return ret
	}
	return circuit.Config{
		General: circuit.GeneralConfig{
			GoLostErrors: s.circuitLostErrors,
		},
		Execution: circuit.ExecutionConfig{
			Timeout:               s.Config.Duration("circuit."+circuitName+".execution.timeout", time.Second).Get(),
			MaxConcurrentRequests: s.Config.Int("circuit."+circuitName+".execution.max_concurrent", 50).Get(),
		},
	}
}

func (s *ServiceCommon) hystrixCloser(circuitName string) hystrix.ConfigureCloser {
	if ret, exists := s.CloserConfig[circuitName]; exists {
		return ret
	}
	return hystrix.ConfigureCloser{
		HalfOpenAttempts:             s.Config.Int("circuit."+circuitName+".closer.halfOpenAttempts", 1).Get(),
		RequiredConcurrentSuccessful: s.Config.Int("circuit."+circuitName+".closer.requiredSuccessful", 1).Get(),
		SleepWindow:                  s.Config.Duration("circuit."+circuitName+".closer.sleepWindow", time.Second*10).Get(),
	}
}

func (s *ServiceCommon) setupHystrix() error {
	statCommandFactory := statsdmetrics.CommandFactory{
		StatSender: s.Statsd.NewSubStatter("circuits"),
	}

	sloTracker := responsetimeslo.Factory{
		CollectorConstructors: []func(circuitName string) responsetimeslo.Collector{statCommandFactory.SLOCollector},
		ConfigConstructor:     []func(circuitName string) responsetimeslo.Config{s.responseSLO},
	}

	hconfig := hystrix.Factory{
		CreateConfigureCloser: []func(circuitName string) hystrix.ConfigureCloser{s.hystrixCloser},
		CreateConfigureOpener: []func(circuitName string) hystrix.ConfigureOpener{s.hystrixOpener},
	}

	rollingStats := rolling.StatFactory{}

	s.Circuit.DefaultCircuitProperties = append(s.Circuit.DefaultCircuitProperties,
		statCommandFactory.CommandProperties, sloTracker.CommandProperties, rollingStats.CreateConfig, s.circuitConfig, hconfig.Configure)

	circuitMetricCollector := statCommandFactory.ConcurrencyCollector(&s.Circuit)
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		go circuitMetricCollector.Start()
		<-s.stopChannel
		if err := circuitMetricCollector.Close(); err != nil {
			s.Log.Log("err", err)
		}
	}()

	return nil
}

func (s *ServiceCommon) setupXRay() error {

	// TODO: Dynamic updating of sample rate
	sampleRate := s.Config.Float(s.Team+"."+s.Service+".xray.sampling", .01)
	s.XRay = &xray.XRay{}
	xrayConfig := xray.Config{
		DefaultName:    s.Team + "-" + s.Service,
		Sampling:       sampleRate.Get(),
		ServiceVersion: s.CodeVersion,
	}

	// If running in ECS, send xray stats to a daemon we assume is on the host at port 2000
	if s.isRunningInECS() {
		if localHostname := s.getEC2InstanceMetadata("local-hostname"); localHostname != "" {
			xrayConfig.DaemonAddr = localHostname + ":2000"
		}
	}
	if err := s.XRay.Configure(xrayConfig); err != nil {
		return err
	}
	sampleRate.Watch(func() {
		if err := s.XRay.PartialConfigure(xray.Config{
			Sampling: sampleRate.Get(),
		}); err != nil {
			s.Log.Log("err", err, "unable to update xray sampling")
		}
	})
	if err := s.XRay.WithPlugin(&ec2.Plugin{}); err != nil {
		// Do not log this error out.  It confuses users
		s.Log.Log("err", err, "unable to setup EC2 plugin.  Your software is probably not running in EC2")
	}
	return nil
}

func (s *ServiceCommon) setupSecrets() error {
	if s.Secrets != nil {
		return nil
	}

	sandstormConf := s.registerSandstormConf(s.Team, s.Service)
	if sandstormConf == nil {
		s.Secrets = &distconf.Distconf{
			Logger: func(key string, err error, msg string) {
				s.Log.Log("key", key, "err", err, msg)
			},
		}
		return nil
	}

	rootConfigs, err := s.rootConfigs()
	if err != nil {
		return err
	}

	s.Secrets = &distconf.Distconf{
		Readers: append(rootConfigs, sandstormConf),
		Logger: func(key string, err error, msg string) {
			s.Log.Log("key", key, "err", err, msg)
		},
	}

	return nil
}

func (s *ServiceCommon) registerSandstormConf(team, service string) *sandyconf.Sandyconf {
	arn := s.Config.Str("sandstorm.arn", "").Get()
	if arn == "" {
		s.Log.Log("No sandstorm ARN set.  Not setting up sandstorm secrets")
		return nil
	}

	mc := sandyconf.ManagerConstructor{}
	sandManager, err := mc.Manager(arn)
	if err != nil {
		s.Log.Log("err", err, "Could not set up sandstorm secrets")
		return nil
	}

	consulFallbackEnv := s.Config.Str("consul.fallback_env", "")

	sandstormEnv := consulFallbackEnv.Get()
	if sandstormEnv == "" {
		sandstormEnv = s.Environment
	}

	sandstormConf := &sandyconf.Sandyconf{
		Team:        team,
		Service:     service,
		Environment: sandstormEnv,
		Manager:     sandManager,
	}
	s.configRefresher.ToRefresh = append(
		s.configRefresher.ToRefresh.(distconf.ComboRefresher), sandstormConf,
	)

	return sandstormConf
}

func (s *ServiceCommon) setupLogging() error {
	s.Config.Logger = func(key string, err error, msg string) {
		s.Log.Log("key", "key", "err", err, "msg", msg)
	}
	if s.Log != nil && s.Log != &SetupLogger {
		// Logging already set.  Don't bother setting it up
		return nil
	}
	// Set a temp logger for errors getting config or secrets setting up logging
	rollbarConfig := rolllog.Config{
		// Get the access token from secrets, not regular config
		AccessToken: s.Secrets.Str("rollbar.access_token", ""),
		SendTimeout: s.Config.Duration("rollbar.send_timeout", time.Second*3),
	}

	clientDefaults := rollbar.DataOptionals{}
	clientDefaults.MergeFrom(&rollbar.DefaultConfiguration)
	clientDefaults.CodeVersion = s.CodeVersion

	rollbarClient := &rollbar.Client{
		Environment:     s.Environment,
		MessageDefaults: &clientDefaults,
	}
	s.rollbarClient = rollbarClient

	rlog := &rolllog.Rolllog{
		OnErr: func(err error) {
			atomic.AddInt64(&s.Stats.RollbarErrors, 1)
		},
		Client:          rollbarClient,
		DefaultErrLevel: rollbar.Error,
		DefaultMsgLevel: rollbar.Info,
	}
	rollbarConfig.Monitor(rlog)
	stdoutLog := fmtlogger.NewLogfmtLogger(os.Stdout, log.Discard)
	gatedStdout := &log.Gate{
		DisabledFlag: 1,
		Logger:       stdoutLog,
	}
	multiLogger := log.MultiLogger([]log.Logger{
		rlog, gatedStdout,
	})
	chanLogger := &log.ChannelLogger{
		Out:    make(chan []interface{}, 64),
		OnFull: log.Discard,
	}
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		log.DrainChannel(multiLogger, chanLogger.Out, s.stopChannel)
	}()
	stdoutEnabled := s.Config.Bool("logging.to_stdout", false)
	stderrEnabled := s.Config.Bool("logging.to_stderr", stdoutEnabled.Get())

	s.PanicLogger = &PanicDumpWriter{
		Output: os.Stderr,
		FlagOn: stderrEnabled,
	}
	watch := func() {
		if stdoutEnabled.Get() {
			gatedStdout.Enable()
		} else {
			gatedStdout.Disable()
		}
	}
	watch()
	stdoutEnabled.Watch(watch)
	normalLogLevel := &StackLogger{
		RootLogger: chanLogger,
	}

	debugLog := &log.Gate{
		DisabledFlag: 1,
		Logger:       normalLogLevel,
	}
	debugEnabled := s.Config.Bool("logging.debug_enabled", false)
	watch2 := func() {
		if debugEnabled.Get() {
			s.Log.Log("Debug logging turned on")
			debugLog.Enable()
		} else {
			s.Log.Log("Debug logging turned off")
			debugLog.Disable()
		}
	}
	watch2()
	debugEnabled.Watch(watch2)
	debugLogLevel := log.NewContext(debugLog).With("level", rollbar.Debug)
	s.Log = &log.ElevatedLog{
		NormalLog: log.ContextLogger{
			Dims:   &s.CtxDimensions,
			Logger: normalLogLevel,
		},
		DebugLog: log.ContextLogger{
			Dims:   &s.CtxDimensions,
			Logger: debugLogLevel,
		},
		ElevateKey: s.ElevateLogKey,
		LogToDebug: func(vals ...interface{}) bool {
			// TODO: Move context.Cancel to debug log
			return false
		},
	}
	s.Log.Debug("Debug log setup complete")
	return nil
}

func sanitizeStatsd(s string) string {
	if len(s) > 32 {
		s = s[0:32]
	}
	return strings.Replace(s, ".", "_", -1)
}

func (s *ServiceCommon) setupStatsd() error {
	if s.Statsd != nil {
		return nil
	}

	var err error
	statsdHostport := s.getStatsdHostPort()
	if statsdHostport == "" {
		s.Log.Log("No statsd hostport setup.  Not sending metrics")
		s.Statsd, err = statsd.NewNoopClient()
		return err
	}
	flushInterval := s.Config.Duration("statsd.flush_interval", time.Second).Get()
	flushBytes := s.Config.Int("statsd.flush_bytes", 512).Get()

	prefix := fmt.Sprintf("%s.%s.%s.%s", sanitizeStatsd(s.Team), sanitizeStatsd(s.Service), sanitizeStatsd(s.Environment), s.getStatsdHost())
	s.statsPrefix = prefix
	s.Statsd, err = statsd.NewBufferedClient(statsdHostport, prefix, flushInterval, int(flushBytes))
	s.Log.Log("addr", statsdHostport, "prefix", prefix, "Statsd setup complete")
	if err != nil {
		return err
	}

	useRegionMetrics := s.Config.Bool("statsd.use_region", false).Get()
	if !useRegionMetrics {
		return nil
	}

	region := s.getDocumentInfo("region")
	if region == "" {
		return nil
	}

	regionStatsdHost := s.Config.Str("statsd.region_host", "statsd.internal.justin.tv:8125").Get()

	regionPrefix := fmt.Sprintf("%s.%s.%s.%s", sanitizeStatsd(s.Team+"-region"), sanitizeStatsd(s.Service), sanitizeStatsd(s.Environment), region)

	regionStats, err := statsd.NewBufferedClient(regionStatsdHost, regionPrefix, flushInterval, int(flushBytes))
	if err != nil {
		return err
	}
	s.Statsd = &multistatsender.MultiStatter{
		Statters: []statsd.Statter{
			regionStats, s.Statsd,
		},
	}

	return nil
}

func (s *ServiceCommon) Datapoints() []*datapoint.Datapoint {
	ret := []*datapoint.Datapoint{
		datapoint.New("code_version", nil, datapoint.NewStringValue(s.CodeVersion), datapoint.Enum, time.Time{}),
	}

	if s.rollbarClient != nil {
		stats := s.rollbarClient.Stats()
		for metricName, stat := range stats.StatMap() {
			ret = append(ret, sfxclient.Cumulative("rollbar."+metricName, map[string]string{"logger": "rollbar"}, stat))
		}
	}

	return ret
}

func mergeMaps(a map[string]string, b map[string]string) map[string]string {
	ret := make(map[string]string, len(a)+len(b))
	for k, v := range a {
		ret[k] = v
	}
	for k, v := range b {
		ret[k] = v
	}
	return ret
}

func (s *ServiceCommon) setupSignalfx() error {
	sfxAuthToken := s.Secrets.Str("signalfx.auth_token", "")
	if sfxAuthToken.Get() == "" {
		s.Log.Log("No signalfx.auth_token.  SignalFx not setup")
		return nil
	}
	defaultDims := map[string]string{
		"team":    s.Team,
		"service": s.Service,
		"env":     s.Environment,
	}
	if s.isRunningInECS() {
		defaultDims["containerId"] = s.Hostname
		defaultDims["host"] = s.getStatsdHost()
	} else {
		defaultDims["host"] = s.Hostname
	}

	reportingDelay := s.Config.Duration("signalfx.reporting_delay", time.Second*10)
	s.SfxSetupConfig.AuthToken = sfxAuthToken.Get()
	s.SfxSetupConfig.ReportingDelay = reportingDelay.Get()
	s.SfxSetupConfig.DefaultDimensions = mergeMaps(defaultDims, s.SfxSetupConfig.DefaultDimensions)
	setupResult, err := sfxstatsd.DefaultSetup(s.SfxSetupConfig)
	if err != nil {
		return err
	}
	setupResult.StatsdTranslator.LogCallback = func(metricName string, metricType string, err error) {
		s.Log.Log("metric", metricName, "type", metricType, "error_val", err.Error(), "missing statsd translation")
	}
	reportingDelay.Watch(func() {
		setupResult.Scheduler.ReportingDelay(reportingDelay.Get())
	})
	sfxAuthToken.Watch(func() {
		// Not thread safe, but meh it'll be fine when used so infrequently
		if asSink, ok := setupResult.Scheduler.Sink.(*sfxclient.HTTPSink); ok {
			asSink.AuthToken = sfxAuthToken.Get()
		}
	})
	setupResult.Scheduler.ErrorHandler = func(err error) error {
		s.Log.Log("err", err, "unable to send signalfx metrics")
		return nil
	}
	s.SfxResult = setupResult
	s.Statsd = &multistatsd.MultiStatter{
		MultiStatSender: multistatsd.MultiStatSender{
			StatSenders: []statsd.StatSender{
				s.Statsd, &setupResult.Statter,
			},
		},
	}
	reportingCtx, onStopCtx := context.WithCancel(context.Background())
	s.SfxResult.Scheduler.AddCallback(s)
	go func() {
		err := s.SfxResult.Scheduler.Schedule(reportingCtx)
		s.Log.Log("err", err)
	}()
	go func() {
		<-s.stopChannel
		onStopCtx()
	}()
	return nil
}

func (s *ServiceCommon) getStatsdHostPort() string {
	// If the service is running in a task in ECS, we assume that it should send stats to a Statsd ECS task on the same
	// ECS host.  Since the Statsd port is bounded to the host's port 8125, we can send stats to the Statsd ECS task
	// by sending to {host-address}:8125.
	if s.isRunningInECS() {
		if privateDns := s.getEC2InstanceMetadata("local-hostname"); privateDns != "" {
			return fmt.Sprintf("%s:8125", privateDns)
		}
	}

	// If the service is running locally or on an EC2 instance, we load the statsd hostport from distconf.
	return s.Config.Str("statsd.hostport", "").Get()
}

func (s *ServiceCommon) getStatsdHost() string {
	// When the service is running in a docker container in ECS, its hostname is the container ID.  Using the container
	// ID as part of a stats prefix is problematic because it changes after every deploy, which will spread stats across
	// many buckets.  This is turn problematic because it'll greatly increase the amount of work Grafana and Graphite
	// will have to do to serve dashboards.
	//
	// To work around this problem, we'll use the instance ID of the instance that the container is running on.
	if s.isRunningInECS() {
		if instanceID := s.getEC2InstanceMetadata("instance-id"); instanceID != "" {
			return sanitizeStatsd(instanceID)
		}
	}

	return sanitizeStatsd(s.Hostname)
}

// getEC2InstanceMetadata loads information about the EC2 instance that the service is running on by querying its
// metadata endpoint.  If the service is not running on an EC2 instance, the request will fail and an empty string
// will be returned.
// http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-metadata.html
func (s *ServiceCommon) getEC2InstanceMetadata(urlSuffx string) string {
	req, err := http.NewRequest("GET", "http://169.254.169.254/latest/meta-data/"+urlSuffx, nil)
	if err != nil {
		s.Log.Log("err", err, "creating request to query instance metadata failed")
		return ""
	}

	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	req = req.WithContext(ctx)

	client := http.DefaultClient
	resp, err := client.Do(req)
	if err != nil {
		s.Log.Log("err", err, "sending request to query instance metadata failed")
		return ""
	}
	if resp.StatusCode != http.StatusOK {
		s.Log.Log("status", resp.StatusCode, "request to query instance metadata failed")
		return ""
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		s.Log.Log("err", err, "reading instance metadata query response failed")
		return ""
	}

	err = resp.Body.Close()
	if err != nil {
		s.Log.Log("err", err, "closing http body failed")
	}

	return string(body)
}

// getDocumentInfo can be used to get things like instance type or region
func (s *ServiceCommon) getDocumentInfo(key string) string {
	req, err := http.NewRequest("GET", "http://169.254.169.254/latest/dynamic/instance-identity/document", nil)
	if err != nil {
		s.Log.Log("err", err, "creating request to query instance metadata failed")
		return ""
	}

	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()
	req = req.WithContext(ctx)

	client := http.DefaultClient
	resp, err := client.Do(req)
	if err != nil {
		s.Log.Log("err", err, "sending request to query instance metadata failed")
		return ""
	}
	if resp.StatusCode != http.StatusOK {
		s.Log.Log("status", resp.StatusCode, "request to query instance metadata failed")
		return ""
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		s.Log.Log("err", err, "reading instance metadata query response failed")
		return ""
	}

	err = resp.Body.Close()
	if err != nil {
		s.Log.Log("err", err, "closing http body failed")
	}

	var decodeInto map[string]string
	if err := json.Unmarshal(body, &decodeInto); err != nil {
		s.Log.Log("err", err, "Decoding document failed")
	}
	return decodeInto[key]
}

func (s *ServiceCommon) isRunningInECS() bool {
	// We can detect whether the service is running on ECS by checking if the AWS_CONTAINER_CREDENTIALS_RELATIVE_URI
	// is defined.
	return s.osGetenv("AWS_CONTAINER_CREDENTIALS_RELATIVE_URI") != ""
}

func (s *ServiceCommon) setupGometrics() error {
	s.ErrorTracker = ErrorTracker{
		Log: s.Log,
	}
	s.gometrics = Gometrics{
		Stats: &StatSender{
			SubStatter:   s.Statsd.NewSubStatter("go"),
			ErrorTracker: &s.ErrorTracker,
		},
		StartTime: time.Now(),
	}

	return nil
}

var startTime = time.Now()

func (s *ServiceCommon) setupBackgroundProcs() {
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		err := s.StatsdTruth.Start()
		s.Log.Log("err", err, "finished sending dummy statsd metrics")
	}()
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		s.gometrics.Start(s.stopChannel, s.Config.Duration("gostats.collect_interval", time.Second*5))
	}()
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		err := s.circuitEventStream.Start()
		s.Log.Log("err", err, "event stream finished")
	}()
}

// Start allows periodic refreshing of config values
func (s *ServiceCommon) Start() error {
	s.setupBackgroundProcsOnce.Do(s.setupBackgroundProcs)
	return s.ConfigCommon.Start()
}

func (s *ServiceCommon) setupExvarHandler() error {
	if s.ExpvarHandler.Exported == nil {
		s.ExpvarHandler.Exported = make(map[string]expvar.Var)
	}
	if s.ExpvarHandler.Logger == nil {
		s.ExpvarHandler.Logger = func(err error) {
			s.Log.Log("err", err)
		}
	}
	s.ExpvarHandler.Exported["env"] = expvar2.EnviromentalVariables()
	s.ExpvarHandler.Exported["conf"] = s.Config.Var()
	s.ExpvarHandler.Exported["dns_cache"] = s.ConfigCommon.DNSCache.Var()
	s.ExpvarHandler.Exported["circuits"] = s.Circuit.Var()
	s.ExpvarHandler.Exported["statsd_prefix"] = expvar.Func(func() interface{} {
		return s.statsPrefix
	})
	if s.rollbarClient != nil {
		s.ExpvarHandler.Exported["rollbar-client"] = s.rollbarClient.Var()
	}
	expvarAwsClient := awsexpvar.Expvar{
		Client: http.DefaultClient,
	}
	s.ExpvarHandler.Exported["aws-metadata"] = expvarAwsClient.Var()
	s.ExpvarHandler.Exported["hostname"] = expvar.Func(func() interface{} {
		return s.Hostname
	})
	s.ExpvarHandler.Exported["untracked_errors"] = s.ErrorTracker.Var()
	s.ExpvarHandler.Exported["uptime"] = expvar.Func(func() interface{} {
		return time.Since(startTime).String()
	})
	if s.SfxResult != nil {
		s.ExpvarHandler.Exported["datapoints"] = s.SfxResult.Scheduler.Var()
	}
	codeVersionExpvar := expvar.String{}
	codeVersionExpvar.Set(s.CodeVersion)
	s.ExpvarHandler.Exported["code_version"] = &codeVersionExpvar
	if s.Secrets != nil {
		s.ExpvarHandler.Exported["secrets"] = expvar.Func(func() interface{} {
			return s.Secrets.Keys()
		})
	}
	s.ExpvarHandler.Exported["xray"] = s.XRay.Var()
	return nil
}

func (s *ServiceCommon) setupDebugPort() error {
	mux := http.NewServeMux()
	d := http.Server{
		Handler: mux,
	}

	mux.Handle("/debug/vars", &s.ExpvarHandler)
	mux.Handle("/debug/pprof/", http.HandlerFunc(pprof.Index))
	mux.Handle("/debug/pprof/cmdline", http.HandlerFunc(pprof.Cmdline))
	mux.Handle("/debug/pprof/profile", http.HandlerFunc(pprof.Profile))
	mux.Handle("/debug/pprof/symbol", http.HandlerFunc(pprof.Symbol))
	mux.Handle("/debug/pprof/trace", http.HandlerFunc(pprof.Trace))

	s.circuitEventStream = &metriceventstream.MetricEventStream{
		Manager: &s.Circuit,
	}

	mux.Handle("/hystrix.stream", s.circuitEventStream)

	var err error
	s.debugListener, err = RetriedListen(context.Background(), s.Config.Str("debug.addr", ":6060").Get(), time.Second*30, time.Second)
	if err != nil {
		return err
	}
	go func() {
		err := d.Serve(s.debugListener)
		s.Log.Log(log.Err, err, "Finished serving on debug port")
	}()
	return nil
}
