package xray

import (
	"context"
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestSimpleCapture(t *testing.T) {
	t.Parallel()
	ctx := context.Background()

	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)

	assert.NoError(t, x.Capture(ctx, "TestService", func(ctx1 context.Context) error {
		ctx = ctx1
		return nil
	}))

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "TestService", s.Name)

	seg := getSegment(ctx)
	assert.Equal(t, seg.TraceID, s.TraceID)
	assert.Equal(t, seg.ID, s.ID)
	assert.Equal(t, seg.StartTime, s.StartTime)
	assert.Equal(t, seg.EndTime, s.EndTime)
	assert.Equal(t, seg, s)
}

func TestErrorCapture(t *testing.T) {
	t.Parallel()
	ctx := context.Background()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)

	err := x.Capture(ctx, "ErrorService", func(ctx1 context.Context) error {
		return stackedError("MyError")
	})

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, err.Error(), s.Cause.Exceptions[0].Message)
	assert.Equal(t, true, s.Fault)
}

func TestPanicCapture(t *testing.T) {
	t.Parallel()
	ctx := context.Background()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)

	var err error
	func() {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%v", p)
			}
		}()

		assert.Error(t, x.Capture(ctx, "PanicService", func(ctx1 context.Context) error {
			panic("MyPanic")
		}))
	}()

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, err.Error(), s.Cause.Exceptions[0].Message)
}

func TestValidAnnotations(t *testing.T) {
	t.Parallel()
	ctx := context.Background()

	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	err := x.Capture(ctx, "Annotations", func(ctx1 context.Context) error {
		ctx = ctx1
		if e := AddAnnotation(ctx, "string", "str"); e != nil {
			return e
		}
		if e := AddAnnotation(ctx, "int", 1); e != nil {
			return e
		}
		if e := AddAnnotation(ctx, "bool", true); e != nil {
			return e
		}
		if e := AddAnnotation(ctx, "float", 1.1); e != nil {
			return e
		}

		return nil
	})
	assert.Nil(t, err)

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "str", s.Annotations["string"])
	assert.Equal(t, 1.0, s.Annotations["int"]) //json encoder turns this into a float64
	assert.Equal(t, 1.1, s.Annotations["float"])
	assert.Equal(t, true, s.Annotations["bool"])
}

func TestInvalidAnnotations(t *testing.T) {
	t.Parallel()
	ctx := context.Background()

	type MyObject struct{}

	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	err := x.Capture(ctx, "BadAnnotations", func(ctx1 context.Context) error {
		ctx = ctx1
		return AddAnnotation(ctx, "Object", &MyObject{})
	})
	assert.Error(t, err)

	_, e := testDaemon.Recv()
	assert.NoError(t, e)

}

func TestSimpleMetadata(t *testing.T) {
	t.Parallel()
	ctx := context.Background()

	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	err := x.Capture(ctx, "MyService", func(ctx1 context.Context) error {
		ctx = ctx1
		if e := AddMetadata(ctx, "string", "str"); e != nil {
			return e
		}
		if e := AddMetadata(ctx, "int", 1); e != nil {
			return e
		}
		if e := AddMetadata(ctx, "bool", true); e != nil {
			return e
		}
		if e := AddMetadata(ctx, "float", 1.1); e != nil {
			return e
		}

		return nil
	})
	assert.Nil(t, err)

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "str", s.Metadata["default"]["string"])
	assert.Equal(t, 1.0, s.Metadata["default"]["int"])
	assert.Equal(t, 1.1, s.Metadata["default"]["float"])
	assert.Equal(t, true, s.Metadata["default"]["bool"])
}

type writeCounter struct {
	count int
}

func (w *writeCounter) Write(p []byte) (n int, err error) {
	w.count++
	return len(p), nil
}

func BenchmarkXRay_Capture(b *testing.B) {
	x := XRay{}
	assert.NoError(b, x.Configure(Config{}))
	emitCount := &writeCounter{}
	x.emitter = emitCount
	ctx := context.Background()
	f := func(context.Context) error {
		return nil
	}
	b.ReportAllocs()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		if err := x.Capture(ctx, "abc", f); err != nil {
			b.Error("unexpected error", err)
		}
	}
	assert.Equal(b, emitCount.count, b.N)
}

func BenchmarkXRay_Capture2(b *testing.B) {
	x := XRay{}
	assert.NoError(b, x.Configure(Config{}))
	emitCount := &writeCounter{}
	x.emitter = emitCount
	ctx := context.Background()
	f := func(context.Context) error {
		return nil
	}

	f2 := func(context.Context) error {
		return x.Capture(ctx, "efg", f)
	}
	b.ReportAllocs()
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		if err := x.Capture(ctx, "abc", f2); err != nil {
			b.Error("unexpected error", err)
		}
	}
	assert.Equal(b, emitCount.count, b.N*2)
}
