package xray

import (
	"context"
	"crypto/rand"
	"fmt"
	"os"
	"sync/atomic"
	"time"
)

// The max size of a segment as defined by AWS is 64kB
// http://docs.aws.amazon.com/xray/latest/devguide/xray-api-segmentdocuments.html
// This is to fit within the max UDP packet size of 65,507 bytes
const maxSegmentSize = 64000

func (x *XRay) newTraceID() string {
	var r [12]byte
	_, err := rand.Read(r[:])
	if err != nil {
		x.onError("unable to generate a trace ID", err)
	}
	return fmt.Sprintf("1-%08x-%02x", time.Now().Unix(), r)
}

func (x *XRay) newSegmentID() string {
	var r [8]byte
	_, err := rand.Read(r[:])
	if err != nil {
		x.onError("unable to generate a trace ID", err)
	}
	return fmt.Sprintf("%02x", r)
}

func (x *XRay) newSegment(ctx context.Context, name string) (context.Context, *segment) {
	atomic.AddInt64(&x.xrayStats.TotalSegments, 1)
	if len(name) > 200 {
		name = name[:200]
	}

	parent := getSegment(ctx)
	seg := &segment{parent: parent}
	seg.Lock()
	defer seg.Unlock()

	if parent == nil {
		seg.TraceID = x.newTraceID()
		seg.sampled = x.sample(seg)
		seg.attachOptionalMetadata(x.optionalMetadata)
		if svcVersion := x.config().ServiceVersion; svcVersion != "" {
			seg.service().Version = svcVersion
		}
	} else {
		parent.Lock()
		parent.rawSubsegments = append(parent.rawSubsegments, seg)
		parent.openSegments++
		parent.Unlock()
	}

	seg.ID = x.newSegmentID()
	seg.Name = name
	seg.StartTime = float64(time.Now().UnixNano()) / float64(time.Second)
	seg.InProgress = true

	return context.WithValue(ctx, contextkey, seg), seg
}

func (seg *segment) close(err error, x *XRay) {
	seg.Lock()

	seg.EndTime = float64(time.Now().UnixNano()) / float64(time.Second)
	seg.InProgress = false

	if err != nil {
		seg.Fault = true
		seg.cause().WorkingDirectory, _ = os.Getwd()
		seg.cause().Exceptions = append(seg.cause().Exceptions, exceptionFromError(err, x.ErrStacks, x))
	}

	seg.Unlock()
	seg.flush(false, x)
}

func (seg *segment) flush(decrement bool, x *XRay) {
	seg.Lock()
	if decrement {
		seg.openSegments--
	}
	shouldFlush := seg.openSegments == 0 && seg.EndTime > 0
	seg.Unlock()

	if shouldFlush {
		if seg.parent == nil {
			x.emit(seg)
		} else {
			seg.parent.flush(true, x)
		}
	}
}

func (seg *segment) attachOptionalMetadata(metadata OptionalMetadata) {
	//Only called within a seg locked code block
	if metadata.IdentityDocument != nil {
		seg.aws().AccountID = metadata.IdentityDocument.AccountID
		seg.aws().ec2().InstanceID = metadata.IdentityDocument.InstanceID
		seg.aws().ec2().AvailabilityZone = metadata.IdentityDocument.AvailabilityZone
	}

	if metadata.ECSContainerName != "" {
		seg.aws().ecs().Container = metadata.ECSContainerName
	}

	if metadata.BeanstalkMetadata != nil {
		seg.aws().elasticBeanstalk().Environment = metadata.BeanstalkMetadata.Environment
		seg.aws().elasticBeanstalk().VersionLabel = metadata.BeanstalkMetadata.VersionLabel
		seg.aws().elasticBeanstalk().DeploymentID = metadata.BeanstalkMetadata.DeploymentID
	}
}

func (seg *segment) addAnnotation(key string, value interface{}) error {
	switch value.(type) {
	case bool, int, uint, float32, float64, string:
	default:
		return fmt.Errorf("failed to add annotation key: %q value: %q to subsegment %q. Value must be of type string, number or boolean", key, value, seg.Name)
	}

	seg.Lock()
	defer seg.Unlock()

	if seg.Annotations == nil {
		seg.Annotations = map[string]interface{}{}
	}
	seg.Annotations[key] = value
	return nil
}

func (seg *segment) addMetadata(key string, value interface{}) error {
	seg.Lock()
	defer seg.Unlock()

	if seg.Metadata == nil {
		seg.Metadata = map[string]map[string]interface{}{}
	}
	if seg.Metadata["default"] == nil {
		seg.Metadata["default"] = map[string]interface{}{}
	}
	seg.Metadata["default"][key] = value
	return nil
}

func (seg *segment) root() *segment {
	if seg.parent == nil {
		return seg
	}
	return seg.parent.root()
}
