/*
Package twitchclient for use with a twitchserver.
*/
package twitchclient

import (
	"crypto/tls"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"math/rand"
	"net"
	"net/http"
	"net/url"
	"strings"
	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/config"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/foundation/xray"
)

func init() {
	rand.Seed(time.Now().UnixNano())
}

// Client is an interface which describes the behavior a standard twitch http client must implement.
type Client interface {
	NewRequest(method string, path string, body io.Reader) (*http.Request, error)
	Do(context.Context, *http.Request, ReqOpts) (*http.Response, error)
	DoNoContent(context.Context, *http.Request, ReqOpts) (*http.Response, error)
	DoJSON(context.Context, interface{}, *http.Request, ReqOpts) (*http.Response, error)
}

// Logger is a logging interface which offers the ability to log, debug log, and do so with
// dimensions attached to the context, like request ID.
type Logger interface {
	DebugCtx(ctx context.Context, params ...interface{})
	Debug(params ...interface{})
	LogCtx(ctx context.Context, params ...interface{})
	Log(params ...interface{})
}

// ClientConf provides the configuration for a new Client
type ClientConf struct {
	// Host (required) configures the client to connect to a specific URI host.
	// If not specified, URIs created by the client will default to use "http://".
	Host string

	// Transport supplies configuration for the client's HTTP transport.
	Transport TransportConf

	// CheckRedirect specifies the policy for handling redirects.
	CheckRedirect func(req *http.Request, via []*http.Request) error

	// Enables tracking of DNS and request timings.
	Stats statsd.Statter

	// Specify a custom stat prefix for DNS timing stats, defaults to "dns"
	DNSStatsPrefix string

	// Enable extra DNS stats, specifically how many IP's are resolved and which is selected
	EnableExtraDNSStats bool

	// Avoid sending the Twitch-Repository header (which is otherwise automatically included).
	// Please set to true when calling 3rd party clients (rollbar, facebook, etc)
	SuppressRepositoryHeader bool

	// The code.justin.tv/chat/timing sub-transaction name, defaults to "twitchhttp"
	TimingXactName string

	// Optional TLS config for making secure requests
	TLSClientConfig *tls.Config

	// Used to modify the BaseRoundTripper used for requests.
	// Wrappers are applied in order: RTW[2](RTW[1](RTW[0](baseRT)))
	RoundTripperWrappers []func(http.RoundTripper) http.RoundTripper

	// Base RoundTripper that makes the http request.
	// By default it is an *http.Transport provided by twitchclient with the current config,
	// but it could be overridden to create mock RoundTripper for tests or similar.
	BaseRoundTripper http.RoundTripper

	// An optional logger. The default implementation prints normal logs to stdout,
	// discards debug logs, and does not log any context dimensions like request ID.
	Logger Logger
	// ctxlog will append logging dimensions to the context using this key. A logger may
	// want to log those context values.
	DimensionKey interface{}
	/* ElevateKey is a key for request contexts. The value will be a boolean which dictates whether or not logs should be elevated to a
	higher log level. Logging implementations may inspect the context for this key and respond accordingly. It must be set to the same
	value as twitchserver. */
	ElevateKey interface{}

	// Used to prefix StatName is requests. One usage of this parameter is to prefix
	// all client stats with your service, e.g. "service.users_service". There
	// shouldn't be a leading period on StatName or a trailing period on StatNamePrefix.
	StatNamePrefix string
}

// TransportConf provides configuration options for the HTTP transport
type TransportConf struct {
	// MaxIdleConnsPerHost controls the maximum number of idle TCP connections
	// that can exist in the connection pool for a specific host. Defaults to
	// http.Transport's default value.
	MaxIdleConnsPerHost int

	// See transport.IdleConnTimeout for details
	// If unset, the DefaultIdleconnTimeout is used
	IdleConnTimeout time.Duration
}

// DefaultIdleConnTimeout is set to 55 seconds
// to be just smaller than the default ALB/ELB idle timeout of 60 seconds
const DefaultIdleConnTimeout = 55 * time.Second

func (tConf *TransportConf) idleConnTimeout() time.Duration {
	if t := tConf.IdleConnTimeout; t > 0 {
		return t
	}
	return DefaultIdleConnTimeout
}

// NewHTTPClient builds a new http.Client using ClientConf,
// with stats, xact, headers and chitin roundtrippers.
// The option ClientConf.Host is ignored (http.Client has no host field).
//
// Use request context to activate stats and add auth headers, i.e.:
//     client := twithhttp.NewHTTPClient(ClientConf{Stats: stats})
//
//     ctx := context.Background()
//     ctx = twitchclient.WithTimingStats(ctx, "statname", 0.2) // track timing stats on "statname.status"
//     ctx = twitchclient.WithTwitchAuthorization(ctx, opts.AuthorizationToken) // add "Twitch-Authorization" header
//     ctx = twitchclient.WithTwitchClientRowID(ctx, opts.ClientRowID) // add "Twitch-Client-Row-ID" header
//     ctx = twitchclient.WithTwitchClientID(ctx, opts.ClientID) // add "Twitch-Client-ID" header
//     req, _ := http.NewRequest("GET", host+"/path", body)
//     req = req.WithContext(ctx)
//
//     response, err := client.Do(req)
//
func NewHTTPClient(conf ClientConf) *http.Client {
	fillConfigDefaults(&conf)
	ctxLog := &ctxlog.Ctxlog{
		CtxDims:       &ctxDimensions{conf.DimensionKey},
		StartingIndex: rand.Int63(),
		ElevateKey:    conf.ElevateKey,
	}
	return resolveHTTPClient(context.Background(), conf, ctxLog)
}

func resolveHTTPTransport(conf ClientConf) *http.Transport {
	netDialer := &net.Dialer{
		Timeout:   30 * time.Second,
		KeepAlive: 30 * time.Second,
	}
	dialWrap := dialerWithDNSStats(netDialer, conf.Stats, conf.DNSStatsPrefix, conf.EnableExtraDNSStats)

	return &http.Transport{
		DialContext:         dialWrap.DialContext,
		Proxy:               http.ProxyFromEnvironment,
		MaxIdleConnsPerHost: conf.Transport.MaxIdleConnsPerHost,
		IdleConnTimeout:     conf.Transport.idleConnTimeout(),
		TLSClientConfig:     conf.TLSClientConfig,
	}
}

func resolveHTTPClient(chitinCtx context.Context, conf ClientConf, ctxLog *ctxlog.Ctxlog) *http.Client {
	// Chitin RoundTripper
	rt, err := chitin.RoundTripper(chitinCtx, conf.BaseRoundTripper) // with ctx.Background() will use req.Context()
	if err != nil {                                                  // new versions of chitin never return an error here
		panic("You are using a very old version of chitin, please upgrade")
	}

	rt = xray.RoundTripper(rt)

	// RoundTripperWrappers
	for _, wrap := range conf.RoundTripperWrappers {
		rt = wrap(rt) // user provided wrappers (i.e. hystrix support)
	}

	// Twitch specific RoundTripper (stats, xact and headers)
	rt = wrapWithTwitchHTTPRoundTripper(rt, conf.Stats, conf.TimingXactName, conf.SuppressRepositoryHeader, ctxLog)

	return &http.Client{
		Transport:     rt,
		CheckRedirect: conf.CheckRedirect,
	}
}

func fillConfigDefaults(conf *ClientConf) {
	if conf.Stats == nil {
		conf.Stats = config.Statsd()
	}
	if conf.BaseRoundTripper == nil {
		conf.BaseRoundTripper = resolveHTTPTransport(*conf)
	}
	if conf.Logger == nil {
		conf.Logger = &defaultLogger{}
	}
	if conf.DimensionKey == nil {
		conf.DimensionKey = new(int)
	}
	if conf.DNSStatsPrefix == "" {
		conf.DNSStatsPrefix = "dns"
	}
}

// NewClient builds a new twitchclient.Client using ClientConf.
// NOTE: You should use NewHTTPClient instead.
func NewClient(conf ClientConf) (Client, error) {
	fillConfigDefaults(&conf)
	hostURL, err := sanitizeHostURL(conf.Host)
	if err != nil {
		return nil, err
	}

	ctxLog := &ctxlog.Ctxlog{
		StartingIndex: rand.Int63(),
		CtxDims:       &ctxDimensions{conf.DimensionKey},
		ElevateKey:    conf.ElevateKey,
	}

	return &client{
		host:   hostURL,
		conf:   &conf,
		logger: conf.Logger,
		ctxLog: ctxLog,
	}, nil
}

type client struct {
	host   *url.URL
	conf   *ClientConf
	logger Logger
	ctxLog *ctxlog.Ctxlog
}

var _ Client = (*client)(nil)

// NewRequest creates an *http.Request using the configured host as the base for the path.
func (c *client) NewRequest(method string, path string, body io.Reader) (*http.Request, error) {
	u, err := url.Parse(path)
	if err != nil {
		return nil, err
	}

	return http.NewRequest(method, c.host.ResolveReference(u).String(), body)
}

func (c *client) modifyReqOpts(opts *ReqOpts) {
	// Default ReqOpt StatSampleRate is 0.1 if StatName was specified
	if opts.StatName != "" && opts.StatSampleRate == 0 {
		opts.StatSampleRate = 0.1
	}

	// Add StatNamePrefix to StatName
	if c.conf.StatNamePrefix != "" {
		opts.StatName = fmt.Sprintf("%s.%s", c.conf.StatNamePrefix, opts.StatName)
	}
}

// do execute a request but leaves error handling and cleanup to the caller.
func (c *client) do(ctx context.Context, req *http.Request, opts ReqOpts) (*http.Response, error) {
	c.modifyReqOpts(&opts)

	// Annotate ctx with opts (used by twitchHTTPRoundTripper)
	ctx = WithReqOpts(ctx, opts)

	// Use provided ctx if req.Context() was not set
	if req.Context() == context.Background() {
		req = req.WithContext(ctx)
	}
	// Make new httpClient using ctx for chitin.RoundTripper
	httpClient := resolveHTTPClient(ctx, *c.conf, c.ctxLog)
	return httpClient.Do(req)
}

// Do executes a requests using the given Context for Trace support
// Do does not close the response body. It is the callers responsibility to do so.
// Do does not perform any error handling on behalf of the user. It is the callers reponsibility to do so.
func (c *client) Do(ctx context.Context, req *http.Request, opts ReqOpts) (*http.Response, error) {
	return c.do(ctx, req, opts)
}

// DoNoContent executes a request using the given Context for Trace support.
// DoNoContent will always close the response body and returns an error on 4xx, 5xx status codes.
// DoNoContent is meant to be used when the caller is not interested in reading the response body.
func (c *client) DoNoContent(ctx context.Context, req *http.Request, opts ReqOpts) (*http.Response, error) {
	resp, err := c.do(ctx, req, opts)
	if err != nil {
		return nil, err
	}

	defer func() {
		if berr := resp.Body.Close(); berr != nil {
			c.logger.Log(fmt.Sprintf("Could not close response body: %v", berr))
		}
	}()

	return resp, parseErrorIfPresent(resp)
}

// DoJSON executes a request, then deserializes the response.
// NOTE: a *twitchclient.Error is returned on 4xx errors, but not on 5xx errors.
// This is problematic because it makes it very difficult for handlers (like Visage) to know if an error
// was caused because of a 5xx backend error or it is a logical error returned by the client.
// For this reason you should not use DoJSON. It is better if you use `Do` instead and always return an error with status.
func (c *client) DoJSON(ctx context.Context, data interface{}, req *http.Request, opts ReqOpts) (*http.Response, error) {
	resp, err := c.do(ctx, req, opts)
	if err != nil {
		return nil, err
	}

	defer func() {
		if berr := resp.Body.Close(); berr != nil {
			c.logger.Log(fmt.Sprintf("Could not close response body: %v", berr))
		}
	}()

	if perr := parseErrorIfPresent(resp); perr != nil {
		return resp, perr
	}

	if resp.StatusCode != http.StatusNoContent {
		err = json.NewDecoder(resp.Body).Decode(data)
		if err != nil {
			return resp, fmt.Errorf("Unable to read response body: %s", err)
		}
	}
	return resp, nil
}

func parseErrorIfPresent(resp *http.Response) error {
	if resp.StatusCode >= 500 {
		body, readAllErr := ioutil.ReadAll(resp.Body)
		if readAllErr != nil {
			return errors.New(resp.Status + ": unable to read response body for more error information")
		}
		return errors.New(resp.Status + ": " + string(body))
	}

	if resp.StatusCode >= 400 {
		return HandleFailedResponse(resp)
	}
	return nil
}

func sanitizeHostURL(host string) (*url.URL, error) {
	if host == "" {
		return nil, fmt.Errorf("Host cannot be blank")
	}
	if !strings.HasPrefix(host, "http") {
		host = fmt.Sprintf("http://%v", host)
	}
	hostURL, err := url.Parse(host)
	if err != nil {
		return nil, err
	}
	return hostURL, nil
}
