package xray

import (
	"context"
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestSimpleCapture(t *testing.T) {
	ctx := context.Background()

	Capture(ctx, "TestService", func(ctx1 context.Context) error {
		ctx = ctx1
		return nil
	})

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "TestService", s.Name)

	seg := getSegment(ctx)
	assert.Equal(t, seg.TraceID, s.TraceID)
	assert.Equal(t, seg.ID, s.ID)
	assert.Equal(t, seg.StartTime, s.StartTime)
	assert.Equal(t, seg.EndTime, s.EndTime)
	assert.Equal(t, seg, s)
}

func TestErrorCapture(t *testing.T) {
	ctx := context.Background()

	err := Capture(ctx, "ErrorService", func(ctx1 context.Context) error {
		return Error("MyError")
	})

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, err.Error(), s.Cause.Exceptions[0].Message)
	assert.Equal(t, true, s.Fault)
}

func TestPanicCapture(t *testing.T) {
	ctx := context.Background()

	var err error
	func() {
		defer func() {
			if p := recover(); p != nil {
				err = fmt.Errorf("%v", p)
			}
		}()

		Capture(ctx, "PanicService", func(ctx1 context.Context) error {
			panic("MyPanic")
		})
	}()

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, err.Error(), s.Cause.Exceptions[0].Message)
}

func TestValidAnnotations(t *testing.T) {
	ctx := context.Background()

	err := Capture(ctx, "Annotations", func(ctx1 context.Context) error {
		ctx = ctx1
		var err MultiError
		if e := AddAnnotation(ctx, "string", "str"); e != nil {
			err = append(err, e)
		}
		if e := AddAnnotation(ctx, "int", 1); e != nil {
			err = append(err, e)
		}
		if e := AddAnnotation(ctx, "bool", true); e != nil {
			err = append(err, e)
		}
		if e := AddAnnotation(ctx, "float", 1.1); e != nil {
			err = append(err, e)
		}

		return err
	})
	assert.Nil(t, err)

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "str", s.Annotations["string"])
	assert.Equal(t, 1.0, s.Annotations["int"]) //json encoder turns this into a float64
	assert.Equal(t, 1.1, s.Annotations["float"])
	assert.Equal(t, true, s.Annotations["bool"])
}

func TestInvalidAnnotations(t *testing.T) {
	ctx := context.Background()

	type MyObject struct{}

	err := Capture(ctx, "BadAnnotations", func(ctx1 context.Context) error {
		ctx = ctx1
		return AddAnnotation(ctx, "Object", &MyObject{})
	})
	assert.Error(t, err)

	_, e := testDaemon.Recv()
	assert.NoError(t, e)

}

func TestSimpleMetadata(t *testing.T) {
	ctx := context.Background()

	err := Capture(ctx, "MyService", func(ctx1 context.Context) error {
		ctx = ctx1
		var err MultiError
		if e := AddMetadata(ctx, "string", "str"); e != nil {
			err = append(err, e)
		}
		if e := AddMetadata(ctx, "int", 1); e != nil {
			err = append(err, e)
		}
		if e := AddMetadata(ctx, "bool", true); e != nil {
			err = append(err, e)
		}
		if e := AddMetadata(ctx, "float", 1.1); e != nil {
			err = append(err, e)
		}

		return err
	})
	assert.Nil(t, err)

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	assert.Equal(t, "str", s.Metadata["default"]["string"])
	assert.Equal(t, 1.0, s.Metadata["default"]["int"])
	assert.Equal(t, 1.1, s.Metadata["default"]["float"])
	assert.Equal(t, true, s.Metadata["default"]["bool"])
}
