package xray

import (
	"encoding/json"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestSplitSegment(t *testing.T) {
	seg := &segment{}
	assert.NoError(t, json.Unmarshal([]byte(testdata), &seg))
	populate(seg)

	assert.True(t, jsonLen(seg) > maxSegmentSize, "Testing segment should be too big")
	for _, b := range encodeIntoPieces(seg, nil) {
		assert.True(t, len(b) < maxSegmentSize, "Emitted segment should be within the allowed size")
	}
}

func jsonLen(v interface{}) int {
	b, _ := json.Marshal(v)
	return len(b)
}

func populate(seg *segment) {
	for _, b := range seg.Subsegments {
		s := &segment{}
		json.Unmarshal(b, &s)
		s.parent = seg
		populate(s)
	}
}

var testdata = `{"trace_id":"1-58e80b8f-bca4523a79f643c6f58e8e3a","id":"8f2b8a1e287b6ad5","name":"visage","start_time":1491602319.7643225,"end_time":1491602321.0354795,"http":{"request":{"method":"POST","url":"/gql","client_ip":"[::1]:65160","user_agent":"Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/57.0.2987.133 Safari/537.36"},"response":{"status":200,"content_length":9902}},"subsegments":[{"id":"53dfa3be34d1cf49","name":"cartman-elb.prod.us-west2.justin.tv","start_time":1491602319.7683766,"end_time":1491602320.1152203,"error":true,"namespace":"remote","http":{"request":{"method":"GET","url":"https://cartman-elb.prod.us-west2.justin.tv/authorization_token?capabilities=cartman%3A%3Aauthenticate_user\u0026key=hmac.key\u0026oauth_scopes="},"response":{"status":401,"content_length":46}}},{"id":"6b04d338792969d4","name":"GraphQL request","start_time":1491602320.1155303,"end_time":1491602321.0351908,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c3df4ef590f7b704","type":"Error","message":"0 errors occurred:\n","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.operationName":"LoggedOutFrontPage"},"metadata":{"default":{"graphql.query":"query LoggedOutFrontPage {\n  featuredStreams(first: 6, geolocation: \"US\", language: \"en\") {\n    title\n    descriptionHTML\n    isScheduled\n    isSponsored\n    imageURL\n    stream {\n      id\n    }\n  }\n  featuredGames: games(first: 12) {\n    name\n    boxArtURL\n    viewersCount\n  }\n  topStreams: streams(first: 6) {\n    ...FrontPageInfo\n  }\n  topPS4Streams: streams(first: 6, platformType: ps4) {\n    ...FrontPageInfo\n  }\n  topXboxStreams: streams(first: 6, platformType: xbox) {\n    ...FrontPageInfo\n  }\n}\n\nfragment FrontPageInfo on Stream {\n  channel { id, name }\n  title\n  viewersCount\n  previewImageURL(width: 300, height: 168)\n}"}},"subsegments":[{"id":"07d904811a7a7c0a","name":"GraphQL field: Query.streams","start_time":1491602320.115596,"end_time":1491602321.0351088,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"7d98d2c196246c1a","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"streams","graphql.type":"Query"},"metadata":{"default":{"graphql.args.first":{"Type":-3,"Text":"6"},"graphql.args.platformType":{"Type":-2,"Text":"xbox"}}},"subsegments":[{"id":"4644c78d69069e8c","name":"jax-internal-production.us-west2.justin.tv","start_time":1491602320.1195264,"end_time":1491602320.730732,"namespace":"remote","http":{"request":{"method":"GET","url":"http://jax-internal-production.us-west2.justin.tv/streams?rails.directory_hidden=false\u0026size=6\u0026xbox_heartbeat=true"},"response":{"status":200,"content_length":7377}}},{"id":"c50461db2947b7da","name":"GraphQL field: Stream.channel","start_time":1491602320.7316484,"end_time":1491602321.0350847,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"4e2304a0694a61fe","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"5400c73f7d59de29","name":"GraphQL field: Channel.name","start_time":1491602320.7317493,"end_time":1491602321.0350707,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"ac28a836c9ec6804","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"e4e144db7487c4d4","name":"GraphQL field: Channel.id","start_time":1491602320.7319448,"end_time":1491602320.7319808,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"13fab902f32cf7b2","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"880f8fd75fd20bac","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7316403,"end_time":1491602320.731698,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"2e38f2dd6af6e523","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"6ac7126947d9c41b","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7316475,"end_time":1491602320.7317028,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"74ac4fe4b7cfe984","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"3c989c8660252014","name":"GraphQL field: Stream.title","start_time":1491602320.7316842,"end_time":1491602320.7317216,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"ccefaf6b89baa908","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"04ad032c51ddf0d0","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7317026,"end_time":1491602320.7317529,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"87b4be97c4915180","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"a1a53d538214f067","name":"GraphQL field: Stream.title","start_time":1491602320.7317133,"end_time":1491602320.731748,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"da6759e1689966ba","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"e134842ca37f191a","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7317336,"end_time":1491602320.7317808,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c221454e409c8e75","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"a1d7f793e70e7b12","name":"GraphQL field: Stream.channel","start_time":1491602320.731747,"end_time":1491602321.035115,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"fb4c5dfa7cd183ed","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"7696cff90f2fbe89","name":"GraphQL field: Channel.name","start_time":1491602320.7318137,"end_time":1491602321.0351,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"99a684c96daae9b8","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"22c7133706a38192","name":"GraphQL field: Channel.id","start_time":1491602320.731879,"end_time":1491602320.7319293,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"86e2e8142ea89013","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"2c3074af96eab234","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7317798,"end_time":1491602320.731834,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"b566d27babc33133","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"d20b11d73b455d45","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.731784,"end_time":1491602320.7318618,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"79372504ea160c4d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"823112d745ab66c5","name":"GraphQL field: Stream.channel","start_time":1491602320.7317858,"end_time":1491602321.0351107,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8a58002a8091fa59","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"3cfafb1c1eeec072","name":"GraphQL field: Channel.name","start_time":1491602320.7319174,"end_time":1491602321.0351024,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"9c607d711864bd7d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"4507dbc219b813f0","name":"GraphQL field: Channel.id","start_time":1491602320.7319698,"end_time":1491602320.7320027,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"82215df7f163ed19","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"0d6ff6c1afaed1e1","name":"GraphQL field: Stream.title","start_time":1491602320.7318037,"end_time":1491602320.7318313,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"584df8ddbc4c818a","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"3605d52526885b51","name":"GraphQL field: Stream.title","start_time":1491602320.731815,"end_time":1491602320.7318547,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"35e8e06372d92087","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"3c4ab8ab0d5d79e5","name":"GraphQL field: Stream.channel","start_time":1491602320.731833,"end_time":1491602321.0351057,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"19518f71df24a450","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"448a1528ba4ea07d","name":"GraphQL field: Channel.name","start_time":1491602320.7318876,"end_time":1491602321.0351012,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"2d60e4cee9aba931","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"5486a89c80e0be80","name":"GraphQL field: Channel.id","start_time":1491602320.7319164,"end_time":1491602320.7319524,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5fc39e28915e6693","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"8a33e7136448ef94","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.731839,"end_time":1491602320.7318683,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8fdc893effcc34fa","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"b9151ae0c4f9a734","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.731862,"end_time":1491602320.7319016,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"9cc50e27ef0a67cc","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"20b3fff6283bcc7d","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.731882,"end_time":1491602320.7319403,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c34d6670d3bc83a6","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"7828bf0e0e5d1c6e","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.731887,"end_time":1491602320.7319243,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f90ce08961a75783","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"f0d45139d9c4d578","name":"GraphQL field: Stream.title","start_time":1491602320.7319078,"end_time":1491602320.7319357,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"1fb60d415417e329","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"498a4ae1a6b030dc","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7319255,"end_time":1491602320.7319522,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"0de334d25a1b8a4a","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"ee2101d39aa2863c","name":"GraphQL field: Stream.channel","start_time":1491602320.7319808,"end_time":1491602321.0351226,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"bf0b6f34e7f696cf","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"d4cfe5fd36f33c85","name":"GraphQL field: Channel.id","start_time":1491602320.7320533,"end_time":1491602320.73209,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"018d3cd11365c35c","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}},{"id":"7deea3b172ed3289","name":"GraphQL field: Channel.name","start_time":1491602320.7320688,"end_time":1491602321.0351195,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"3ed1e0b4a1afb642","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}}]},{"id":"cd577d48727b1afb","name":"GraphQL field: Stream.channel","start_time":1491602320.7319875,"end_time":1491602321.035118,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5a3d6dfcf0e974f7","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"086cfd612572871a","name":"GraphQL field: Channel.name","start_time":1491602320.7320411,"end_time":1491602321.0351274,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e04c77ad029a7769","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"00be0ce62464abd5","name":"GraphQL field: Channel.id","start_time":1491602320.7320476,"end_time":1491602320.7320962,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"fedd0db50cc27074","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"93ab8edc5ac3fa8a","name":"GraphQL field: Stream.title","start_time":1491602320.7319882,"end_time":1491602320.7320216,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5e5eca324934a02b","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"049883cbd182f18d","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7319891,"end_time":1491602320.732017,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"7ca721071d3c1046","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}}]},{"id":"889d75d5a1f31777","name":"GraphQL field: Query.games","start_time":1491602320.115636,"end_time":1491602320.253123,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"3db42b035c6505d5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"games","graphql.type":"Query"},"metadata":{"default":{"graphql.args.first":{"Type":-3,"Text":"12"}}},"subsegments":[{"id":"1c9c0590b8e0217a","name":"discovery-production.prod.us-west2.twitch.tv:9292","start_time":1491602320.123476,"end_time":1491602320.2230046,"namespace":"remote","http":{"request":{"method":"GET","url":"http://discovery-production.prod.us-west2.twitch.tv:9292/top?limit=12"},"response":{"status":200,"content_length":19483}}},{"id":"4289ab8de2facd48","name":"GraphQL field: Game.viewersCount","start_time":1491602320.25245,"end_time":1491602320.2527297,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"adbe0470367eef49","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"60649c924ec5b7fc","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2524521,"end_time":1491602320.252827,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8960d776bbf4bdff","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"45abf3c9cb4a0f4d","name":"GraphQL field: Game.viewersCount","start_time":1491602320.252498,"end_time":1491602320.2528906,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"058c3eabea2fcdfe","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"f1913e65028d5094","name":"GraphQL field: Game.name","start_time":1491602320.2524831,"end_time":1491602320.252891,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e3575da82361471d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"41c8e676508d15cd","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2524884,"end_time":1491602320.2525244,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"037a5b34a71ac0bf","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"f25b7735760d1b84","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2525904,"end_time":1491602320.2530289,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"242e24587b85e1c5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"4a2e41579b162004","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2525606,"end_time":1491602320.2529292,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c0ca38613e246eda","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"f986ddadafa588e6","name":"GraphQL field: Game.name","start_time":1491602320.2525082,"end_time":1491602320.2528574,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"672c08f3c8ad43d9","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"9389c79aaa99d5ce","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2525613,"end_time":1491602320.2529123,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"35a0988b4cc8a407","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"d7d71900987332d6","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2525647,"end_time":1491602320.2528243,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"77fdcf2b7bf0aba0","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"72f597ff6e6fdd64","name":"GraphQL field: Game.name","start_time":1491602320.252583,"end_time":1491602320.2528799,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"54e72e501a945341","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"cf1871518ffa6722","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2525876,"end_time":1491602320.2528734,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"2122915b3cde85f2","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"82acdf8b183634db","name":"GraphQL field: Game.name","start_time":1491602320.252618,"end_time":1491602320.2529273,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"aa08748837246542","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"ac1f08a63a1fb3d1","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.252597,"end_time":1491602320.2530153,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5574600c7255be66","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"3f864f4149533ad1","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2526119,"end_time":1491602320.2528365,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"94068033a7abe07f","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"a48733ae66932ef3","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.25262,"end_time":1491602320.2531064,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"08992b1bcc8964fa","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"7ee5decf8c47647b","name":"GraphQL field: Game.name","start_time":1491602320.2528052,"end_time":1491602320.2528324,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"28f01c656a305881","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"8e0473e9d276a456","name":"GraphQL field: Game.name","start_time":1491602320.2526839,"end_time":1491602320.2528784,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"9841ceaf5d0548d5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"2b506b1f739909df","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2526574,"end_time":1491602320.2529123,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"d6ef48310e59b5a9","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"01b48168c1a2d662","name":"GraphQL field: Game.name","start_time":1491602320.2526588,"end_time":1491602320.252874,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8041eea9a02d1569","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"a646543f7280b5c5","name":"GraphQL field: Game.name","start_time":1491602320.2526448,"end_time":1491602320.2529082,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"2a525926b6158a87","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"e9722648a2d8ae78","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2526515,"end_time":1491602320.253045,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"409d5d42659a91e0","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"75a1fcc28779c457","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2528434,"end_time":1491602320.252979,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c634cffe36e60f6d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"0ba0ef792d6bb80c","name":"GraphQL field: Game.name","start_time":1491602320.2526865,"end_time":1491602320.2529492,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"6f7f585bede24fc7","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"058f5ccef0d01a46","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2527454,"end_time":1491602320.2529843,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c42fc28822345426","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"026c2b56c1219e19","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2526953,"end_time":1491602320.2530565,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"a42c8a3bede622da","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"dd8b6e6f96909dd3","name":"GraphQL field: Game.viewersCount","start_time":1491602320.252699,"end_time":1491602320.2529137,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"05bc9d6fc3a8c038","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"4450f13fdde80900","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2527235,"end_time":1491602320.2529962,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"120d3e447390f573","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"08c4f5b75829d2b8","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2527187,"end_time":1491602320.2530844,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e65f833af7a6cd1e","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"e594b9bf531d7fc3","name":"GraphQL field: Game.name","start_time":1491602320.252761,"end_time":1491602320.2529438,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f1d157e44925dd39","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"33238550ade59aac","name":"GraphQL field: Game.name","start_time":1491602320.2527604,"end_time":1491602320.2529452,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"748695347b2c16a1","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"55ff1034964dad9d","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2527688,"end_time":1491602320.2531157,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"a4121dff01606132","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"4b8cc8d6c4859b54","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.2527754,"end_time":1491602320.2529926,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"aacd92d3fc166b48","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}},{"id":"5cba985d30c5f9e1","name":"GraphQL field: Game.viewersCount","start_time":1491602320.2528262,"end_time":1491602320.2528567,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"936a330bfc6a9844","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Game"}},{"id":"1057e18404522ea5","name":"GraphQL field: Game.name","start_time":1491602320.25288,"end_time":1491602320.252928,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"b4ab52d89d770542","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Game"}},{"id":"08f0d7a3948c047a","name":"GraphQL field: Game.boxArtURL","start_time":1491602320.252892,"end_time":1491602320.2530763,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"24d27bc474948b05","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"boxArtURL","graphql.type":"Game"}}]},{"id":"89a708d7fb7101c4","name":"GraphQL field: Query.streams","start_time":1491602320.1156442,"end_time":1491602321.0351453,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"29b7a099aa895a54","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"streams","graphql.type":"Query"},"metadata":{"default":{"graphql.args.first":{"Type":-3,"Text":"6"}}},"subsegments":[{"id":"4459b64eb4677086","name":"jax-internal-production.us-west2.justin.tv","start_time":1491602320.1194706,"end_time":1491602320.7381518,"namespace":"remote","http":{"request":{"method":"GET","url":"http://jax-internal-production.us-west2.justin.tv/streams?rails.directory_hidden=false\u0026size=6"},"response":{"status":200,"content_length":7676}}},{"id":"7b66af1e85ff1b25","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7390127,"end_time":1491602320.7391002,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"ce672ed562a84991","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"618ff21be6435c49","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7390127,"end_time":1491602320.7392774,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"3313427bc67ae570","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"5bfb68804d46c237","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7390513,"end_time":1491602320.7391086,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"4b7c18719a44f101","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"477a98d599e5c77a","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7390535,"end_time":1491602320.7391183,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"85d804cd9a8364de","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"bfb18c1dc53a055b","name":"GraphQL field: Stream.title","start_time":1491602320.7390735,"end_time":1491602320.7391186,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5714958a766dcd48","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"384358b4b6452b33","name":"GraphQL field: Stream.channel","start_time":1491602320.7390904,"end_time":1491602321.03513,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"401b8fa63182c198","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"1290e6af40d49aa1","name":"GraphQL field: Channel.name","start_time":1491602320.7391465,"end_time":1491602321.035128,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e3f93f714c6578d5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"ae1591d7f2b650b9","name":"GraphQL field: Channel.id","start_time":1491602320.7392879,"end_time":1491602320.7393234,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"3ca58dedb8325130","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"64d286e0a7908f27","name":"GraphQL field: Stream.channel","start_time":1491602320.739179,"end_time":1491602321.0350769,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"7e06c180577414f4","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"70afd6ee49f10ed8","name":"GraphQL field: Channel.name","start_time":1491602320.73924,"end_time":1491602321.0350633,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f0cb1b9762784c73","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"f3cc7025822b8a89","name":"GraphQL field: Channel.id","start_time":1491602320.7394836,"end_time":1491602320.7395205,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c9e0e4da4f98be32","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"e73088babe13c0b6","name":"GraphQL field: Stream.channel","start_time":1491602320.7391934,"end_time":1491602321.0350544,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c77c69c37f0ab1a6","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"fc178a686a6fa81a","name":"GraphQL field: Channel.name","start_time":1491602320.7392669,"end_time":1491602321.035022,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"a319a881c5040ee9","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"78a23a4917825854","name":"GraphQL field: Channel.id","start_time":1491602320.7393265,"end_time":1491602320.739361,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"b43f1f4bf7478b31","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"3c537a896d48332a","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.739191,"end_time":1491602320.73923,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"7563d63271c63aca","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"3b0d2295a1231cae","name":"GraphQL field: Stream.title","start_time":1491602320.7392027,"end_time":1491602320.73923,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"476b77240c05509c","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"f913a92d7b0a3eb3","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.739209,"end_time":1491602320.7392437,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"9e44589666dce97e","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"2ec931082cc5bd81","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7392545,"end_time":1491602320.7392964,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"216ff6633650665d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"0615f307036d20f7","name":"GraphQL field: Stream.title","start_time":1491602320.7393067,"end_time":1491602320.7393372,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"03ed370834b382b5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"569a16902ba0ea8b","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7393172,"end_time":1491602320.739357,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"bbf2a05ddc926186","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"15d5ddd948b33a82","name":"GraphQL field: Stream.channel","start_time":1491602320.7393243,"end_time":1491602321.035079,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"0169b2d8f3f5575f","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"89337ed60e4ee450","name":"GraphQL field: Channel.name","start_time":1491602320.73938,"end_time":1491602321.0350945,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e7f416679a0d1f5c","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"d99e175037fbca8c","name":"GraphQL field: Channel.id","start_time":1491602320.7395148,"end_time":1491602320.7395513,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"d11b91b88e14c46e","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"f65a074fa6fa8935","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7393587,"end_time":1491602320.7393947,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"cb252c0f1d506a50","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"500cdebcad36b97b","name":"GraphQL field: Stream.title","start_time":1491602320.7393758,"end_time":1491602320.7394042,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5032919a061e7d47","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"1a56e0bdc8876e18","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7393963,"end_time":1491602320.7394266,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"340154a993068b32","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"e93209972e00ded6","name":"GraphQL field: Stream.channel","start_time":1491602320.7394059,"end_time":1491602321.0350888,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"968e25f061581dc3","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"f82399286c06b340","name":"GraphQL field: Channel.name","start_time":1491602320.7395039,"end_time":1491602321.0350776,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c62dbc944f7ab734","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"bb90637d1a3c49db","name":"GraphQL field: Channel.id","start_time":1491602320.7395718,"end_time":1491602320.7395914,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"055a22fbb38b6630","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"fdbe1e467817cb3c","name":"GraphQL field: Stream.channel","start_time":1491602320.7394269,"end_time":1491602321.0351162,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"189ee5397103b882","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"a13e8245669b5831","name":"GraphQL field: Channel.name","start_time":1491602320.7394834,"end_time":1491602321.0351007,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"040e3b738bb93f92","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"073bd9fbd728bf0f","name":"GraphQL field: Channel.id","start_time":1491602320.7395222,"end_time":1491602320.7395544,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"7a4c51f42e6c5025","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"3fd146369bdf01aa","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7394314,"end_time":1491602320.7394593,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8abafcbb8ff3324d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"bd9377d3e6ffdd3c","name":"GraphQL field: Stream.title","start_time":1491602320.7394435,"end_time":1491602320.7394712,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"ad0f4072b63efa13","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"6f5f6ccf57108e2f","name":"GraphQL field: Stream.title","start_time":1491602320.7395837,"end_time":1491602320.739616,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c9362c025e82eaa7","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"1bf56578db396447","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7395403,"end_time":1491602320.7395656,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"b8834156c80ad84e","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}}]},{"id":"77998eadb0bc3b37","name":"GraphQL field: Query.streams","start_time":1491602320.115636,"end_time":1491602321.0350616,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"7b194b12f473cab0","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"streams","graphql.type":"Query"},"metadata":{"default":{"graphql.args.first":{"Type":-3,"Text":"6"},"graphql.args.platformType":{"Type":-2,"Text":"ps4"}}},"subsegments":[{"id":"3c41187607db3561","name":"jax-internal-production.us-west2.justin.tv","start_time":1491602320.1194715,"end_time":1491602320.739768,"namespace":"remote","http":{"request":{"method":"GET","url":"http://jax-internal-production.us-west2.justin.tv/streams?playstation.sce_platform=PS4\u0026rails.directory_hidden=false\u0026size=6"},"response":{"status":200,"content_length":12091}}},{"id":"9e5c22575eca88a0","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7409616,"end_time":1491602320.7409852,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"58c9a6444dbdb7a9","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"d9c8256fce1c664b","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.740989,"end_time":1491602320.741023,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8edadf1a30881bf1","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"5dae3bb6f5b21732","name":"GraphQL field: Stream.channel","start_time":1491602320.7410169,"end_time":1491602321.03506,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f84c98827e97e52c","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"134ffaec973fd596","name":"GraphQL field: Channel.name","start_time":1491602320.7419047,"end_time":1491602321.035039,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"37a13dcda47b78eb","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"a1bf3dd2ac6159fe","name":"GraphQL field: Channel.id","start_time":1491602320.7420812,"end_time":1491602320.7421153,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e1c30139769ad107","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"3b69e8d3d4226a42","name":"GraphQL field: Stream.title","start_time":1491602320.7410207,"end_time":1491602320.7418242,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"d6828ae3cd4d8189","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"d15d119958607c7b","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.741022,"end_time":1491602320.74228,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5ce6ac42950d59d5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"a9eac235afd2b25a","name":"GraphQL field: Stream.title","start_time":1491602320.741029,"end_time":1491602320.7426713,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"bdd7733670ab528d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"d6bca8cc2190537c","name":"GraphQL field: Stream.channel","start_time":1491602320.7410333,"end_time":1491602321.0351014,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"a0765cafcd32dbe8","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"6c76fd1658cdb872","name":"GraphQL field: Channel.name","start_time":1491602320.742212,"end_time":1491602321.0350864,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"bdab25f4ff757ab9","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"0def3515fa5b96a8","name":"GraphQL field: Channel.id","start_time":1491602320.7423112,"end_time":1491602320.7423573,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"2194f7a95c615fff","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"547128a44e09347a","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7410395,"end_time":1491602320.7417016,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"72fcf75d37964fa4","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"97a8694af2ca6220","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7416394,"end_time":1491602320.7416868,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"d904b84d81dc4e23","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"f7f7958d53d90034","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7416694,"end_time":1491602320.7417386,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"bec614a8458a667f","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"095e135e7c42c7bb","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7417636,"end_time":1491602320.741835,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"eb66ed51957c8227","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"f862dd940940bb12","name":"GraphQL field: Stream.channel","start_time":1491602320.7419999,"end_time":1491602321.0350459,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"21355193ccb7b745","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"ff3212148976e5db","name":"GraphQL field: Channel.name","start_time":1491602320.742244,"end_time":1491602321.035003,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"0cdb4ecfb091c65c","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"ca547d7217b23fca","name":"GraphQL field: Channel.id","start_time":1491602320.7425582,"end_time":1491602320.7426128,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"87b4b373810bb1b0","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"65cb228bfc39e856","name":"GraphQL field: Stream.title","start_time":1491602320.742046,"end_time":1491602320.74209,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"9ebbb12a4a1d1d45","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"bf9d6c81203b196a","name":"GraphQL field: Stream.previewImageURL","start_time":1491602320.7421532,"end_time":1491602320.7422142,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"978db3a8d669b03a","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"previewImageURL","graphql.type":"Stream"},"metadata":{"default":{"graphql.args.height":{"Type":-3,"Text":"168"},"graphql.args.width":{"Type":-3,"Text":"300"}}}},{"id":"5101a160ada20344","name":"GraphQL field: Stream.channel","start_time":1491602320.742173,"end_time":1491602321.0350711,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c6a7d95e3fad3493","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"529905d10be15549","name":"GraphQL field: Channel.name","start_time":1491602320.7422338,"end_time":1491602321.0350332,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"33c3d12a3a9feec6","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"fcf8998c4b931b6c","name":"GraphQL field: Channel.id","start_time":1491602320.7423368,"end_time":1491602320.7423787,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"a48e759f31d44635","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"99c7730da9f911e3","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7422702,"end_time":1491602320.742309,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"0e2eb39b4c1a4262","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"00399e782cdc495f","name":"GraphQL field: Stream.channel","start_time":1491602320.742303,"end_time":1491602321.0350847,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f9610c1786a74fa1","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"850c38734b435eb1","name":"GraphQL field: Channel.name","start_time":1491602320.7424018,"end_time":1491602321.0350628,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"0953efbd821cc914","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"b2e3666e58fa041a","name":"GraphQL field: Channel.id","start_time":1491602320.7425272,"end_time":1491602320.74257,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"66c1900fa7661539","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"59308ca7804e5dc9","name":"GraphQL field: Stream.channel","start_time":1491602320.7423613,"end_time":1491602321.0350006,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"df7ae7aa17378d1c","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"channel","graphql.type":"Stream"},"subsegments":[{"id":"0a8825b127406234","name":"GraphQL field: Channel.name","start_time":1491602320.7426312,"end_time":1491602321.035046,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"561a13146a039e98","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"name","graphql.type":"Channel"}},{"id":"2f2c4e3ffb6e1f8f","name":"GraphQL field: Channel.id","start_time":1491602320.7425823,"end_time":1491602320.7426267,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5371b45ee686d69b","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Channel"}}]},{"id":"a89f92d849ebe55a","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7423985,"end_time":1491602320.7424378,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"d2672c32841bece5","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"86c8967c5cd6a19e","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7424192,"end_time":1491602320.742461,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"07c9baefec4d71f8","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"7f88d9f6595f867a","name":"GraphQL field: Stream.viewersCount","start_time":1491602320.7424395,"end_time":1491602320.7424788,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"8b12ac4f63000c08","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"viewersCount","graphql.type":"Stream"}},{"id":"12ee892a22551c63","name":"GraphQL field: Stream.title","start_time":1491602320.7425225,"end_time":1491602320.7425897,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"377788ed1869ed5b","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"fdb0c82ec58bd9a6","name":"GraphQL field: Stream.title","start_time":1491602320.7425315,"end_time":1491602320.7425694,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"242507abd80fe18d","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}},{"id":"8d7afad33a84fa45","name":"GraphQL field: Stream.title","start_time":1491602320.7426093,"end_time":1491602320.742653,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"c61585dc4621cdf1","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"title","graphql.type":"Stream"}}]},{"id":"a85e9cc723f3cb69","name":"GraphQL field: Query.featuredStreams","start_time":1491602320.1157362,"end_time":1491602320.7345223,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"95f7f45722aafed8","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"featuredStreams","graphql.type":"Query"},"metadata":{"default":{"graphql.args.first":{"Type":-3,"Text":"6"},"graphql.args.geolocation":{"Type":-6,"Text":"\"US\""},"graphql.args.language":{"Type":-6,"Text":"\"en\""}}},"subsegments":[{"id":"5d7799efc5d0b3d2","name":"api.twitch.tv","start_time":1491602320.1282263,"end_time":1491602320.5444798,"namespace":"remote","http":{"request":{"method":"GET","url":"http://internal-alb-internal-nginx-api-1854537848.us-west-2.elb.amazonaws.com/kraken/streams/featured?geo=US\u0026lang=en\u0026limit=6\u0026offset=0\u0026on_site=1"},"response":{"status":200}}},{"id":"25421066d1a66c6a","name":"GraphQL field: FeaturedStream.stream","start_time":1491602320.5471425,"end_time":1491602320.7345035,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"088d2691153eeba4","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"stream","graphql.type":"FeaturedStream"},"subsegments":[{"id":"d3189d23d5c7d5e0","name":"GraphQL field: Stream.id","start_time":1491602320.5473025,"end_time":1491602320.7344954,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"54e842734514fa71","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Stream"}}]},{"id":"a5573758903ada05","name":"GraphQL field: FeaturedStream.stream","start_time":1491602320.547146,"end_time":1491602320.7345042,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"4e93c93021036b3b","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"stream","graphql.type":"FeaturedStream"},"subsegments":[{"id":"41e0c849d8827c48","name":"GraphQL field: Stream.id","start_time":1491602320.547296,"end_time":1491602320.7345057,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f2a066cdab23b044","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Stream"}}]},{"id":"119d5d958d8ba388","name":"GraphQL field: FeaturedStream.stream","start_time":1491602320.5471427,"end_time":1491602320.734511,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"e79af7c72925ff55","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"stream","graphql.type":"FeaturedStream"},"subsegments":[{"id":"de592a9f8ddc83af","name":"GraphQL field: Stream.id","start_time":1491602320.5473118,"end_time":1491602320.7345114,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"4b887d17dd50d468","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Stream"}}]},{"id":"13632608151bafe1","name":"GraphQL field: FeaturedStream.stream","start_time":1491602320.547154,"end_time":1491602320.7345195,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"24693cbca45d3082","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"stream","graphql.type":"FeaturedStream"},"subsegments":[{"id":"5a05f48bb9f76373","name":"GraphQL field: Stream.id","start_time":1491602320.5473018,"end_time":1491602320.7345045,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"56a7d7628726049f","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Stream"}}]},{"id":"ef6c5dd566e9e4b2","name":"GraphQL field: FeaturedStream.stream","start_time":1491602320.5472019,"end_time":1491602320.7344692,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"5fcf2ed0e3e9c8a9","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"stream","graphql.type":"FeaturedStream"},"subsegments":[{"id":"50e5f7ebd623f270","name":"GraphQL field: Stream.id","start_time":1491602320.547244,"end_time":1491602320.7344723,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"988cd506df900954","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Stream"}}]},{"id":"e183e791f5fb31d4","name":"GraphQL field: FeaturedStream.stream","start_time":1491602320.5472105,"end_time":1491602320.7344842,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"f868ef50bb434334","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"stream","graphql.type":"FeaturedStream"},"subsegments":[{"id":"e2b965e7415393f5","name":"GraphQL field: Stream.id","start_time":1491602320.547249,"end_time":1491602320.7344635,"fault":true,"cause":{"working_directory":"/Users/gamble/go/src/code.justin.tv/edge/visage","exceptions":[{"id":"36f0f03ca463d020","type":"Error","message":"\u003cnil\u003e","stack":[{"path":"code.justin.tv/edge/visage/vendor/code.justin.tv/foundation/xray/capture.go","line":18,"label":"Capture"}]}]},"annotations":{"graphql.field":"id","graphql.type":"Stream"}}]}]}]},{"id":"c462ea15a870e016","name":"jax-internal-production.us-west2.justin.tv","start_time":1491602320.5691824,"end_time":1491602320.7339299,"namespace":"remote","http":{"request":{"method":"POST","url":"http://jax-internal-production.us-west2.justin.tv/stream/list?"},"response":{"status":200,"content_length":7936}}},{"id":"0b5489da558095f0","name":"users-service.dev.us-west2.twitch.tv","start_time":1491602320.755803,"end_time":1491602321.0039482,"namespace":"remote","http":{"request":{"method":"GET","url":"http://users-service.dev.us-west2.twitch.tv/channels?id=150684601\u0026id=84030521\u0026id=142210011\u0026id=102742114\u0026id=127561703\u0026id=150808792\u0026id=87204022\u0026id=14836307\u0026id=12875057\u0026id=43131877\u0026id=23161357\u0026id=26490481\u0026id=73498909\u0026id=116234094\u0026id=90427687\u0026id=48359499\u0026id=86526922\u0026id=71920645"},"response":{"status":200,"content_length":27760}}}]}`
