package paramstoreconf_test

import (
	"os"
	"testing"

	"code.justin.tv/hygienic/distconf"
	"code.justin.tv/hygienic/paramstoreconf"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ssm"
	"github.com/cep21/circuit"
)

func getProfile() string {
	return os.Getenv("AWS_PROFILE")
}

func getRegion() string {
	return os.Getenv("AWS_DEFAULT_REGION")
}

func TestParameterStoreConfigurationNormal(t *testing.T) {
	profile := getProfile()
	if profile == "" {
		t.Skip("missing AWS_PROFILE")
	}
	region := getRegion()
	if region == "" {
		t.Skip("missing AWS_DEFAULT_REGION")
	}
	opts := session.Options{
		SharedConfigState: session.SharedConfigEnable,
		Profile:           profile,
		Config: aws.Config{
			Region: &region,
		},
	}
	awsSessionForRole, err := session.NewSessionWithOptions(opts)
	if err != nil {
		t.Fatal("unable to get AWS session")
	}

	client := ssm.New(awsSessionForRole)
	m := circuit.Manager{}
	p := &paramstoreconf.ParameterStoreConfiguration{
		Prefix:  "testing",
		Team:    "twitch",
		SSM:     client,
		Circuit: m.MustCreateCircuit("-testing-"),
	}
	d := distconf.Distconf{
		Readers: []distconf.Reader{
			p,
		},
	}
	s := d.Str("normal_string", "")
	if s.Get() != "is_set" {
		t.Fatal("string not set: TestParameterStoreConfiguration")
	}

	s = d.Str("enc_string", "")
	if s.Get() != "" {
		t.Fatal("Able to read encrypted strings")
	}
}

func TestParameterStoreConfigurationSecrets(t *testing.T) {
	profile := getProfile()
	if profile == "" {
		t.Skip("missing AWS_PROFILE")
	}
	region := getRegion()
	if region == "" {
		t.Skip("missing AWS_DEFAULT_REGION")
	}
	opts := session.Options{
		SharedConfigState: session.SharedConfigEnable,
		Profile:           profile,
		Config: aws.Config{
			Region: &region,
		},
	}
	awsSessionForRole, err := session.NewSessionWithOptions(opts)
	if err != nil {
		t.Fatal("unable to get AWS session")
	}

	client := ssm.New(awsSessionForRole)
	m := circuit.Manager{}
	p := &paramstoreconf.ParameterStoreConfiguration{
		Prefix:         "testing",
		Team:           "twitch",
		SSM:            client,
		AllowEncrypted: true,
		Circuit:        m.MustCreateCircuit("-testing-"),
	}
	d := distconf.Distconf{
		Readers: []distconf.Reader{
			p,
		},
	}
	s := d.Str("normal_string", "")
	if s.Get() != "is_set" {
		t.Fatal("string not set: TestParameterStoreConfiguration")
	}

	s = d.Str("enc_string", "")
	if s.Get() != "is_set2" {
		t.Fatal("Unable to read encrypted strings", s.Get())
	}
}
