package main

import (
	"bytes"
	"context"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path"
	"regexp"

	"code.justin.tv/release/trace/internal/amzncorp"
)

func main() {
	var (
		cookieFilename = flag.String("cookiejar", path.Join(os.Getenv("HOME"), ".midway/cookie"),
			"Path to Midway cookie file generated by mwinit")
		accountID = flag.String("account", "", "AWS Account ID")
		roleName  = flag.String("role", "", "Name of IAM role to assume")
	)
	flag.Parse()

	ctx := context.Background()

	log.SetFlags(0)

	if len(flag.Args()) < 1 {
		log.Fatalf("no command listed")
	}
	cmdName := flag.Args()[0]
	cmdArgs := flag.Args()[1:]

	if !regexp.MustCompile(`^[0-9]{12}$`).MatchString(*accountID) {
		log.Fatalf("account must be a 12-digit number")
	}
	if *roleName == "" {
		log.Fatalf("role must be set")
	}

	cookieFile, err := ioutil.ReadFile(*cookieFilename)
	if err != nil {
		log.Fatalf("cookiejar %q could not be opened: %v", *cookieFilename, err)
	}

	client, err := amzncorp.MidwayClient(bytes.NewReader(cookieFile))
	if err != nil {
		log.Fatalf("MidwayClient: %v", err)
	}

	creds := &amzncorp.IsengardCredentials{
		MidwayClient: client,
		AWSAccountID: *accountID,
		IAMRoleName:  *roleName,
	}

	v, err := creds.Retrieve()
	if err != nil {
		log.Fatalf("Retrieve: %v", err)
	}

	cmd := exec.CommandContext(ctx, cmdName, cmdArgs...)
	cmd.Env = append(os.Environ(),
		fmt.Sprintf("AWS_ACCESS_KEY_ID=%s", v.AccessKeyID),
		fmt.Sprintf("AWS_SECRET_ACCESS_KEY=%s", v.SecretAccessKey),
		fmt.Sprintf("AWS_SESSION_TOKEN=%s", v.SessionToken))
	cmd.Stdin, cmd.Stdout, cmd.Stderr = os.Stdin, os.Stdout, os.Stderr

	err = cmd.Run()
	if err != nil {
		log.Fatalf("os/exec.Cmd.Run: %v", err)
	}
}
