package htmlreport

import (
	"bytes"
	"html/template"
	"path"
	"sort"
	"strings"

	"code.justin.tv/release/trace/analysis/render"
	"code.justin.tv/release/trace/api/report_v1"
)

type URIGenerator interface {
	URIFor(partial string) string
	URIForCallRef(ref *report_v1.CallRef) string
	URIForProgramRef(ref *report_v1.ProgramRef) string
}

func baseTemplate(routes URIGenerator) (*template.Template, error) {
	t := template.New("")

	t = t.Funcs(template.FuncMap{
		"URLFor": func(subpath string) string {
			return routes.URIFor(subpath)
		},
		"URLForCallRef": func(ref *report_v1.CallRef) string {
			if ref == nil {
				return ""
			}
			return routes.URIForCallRef(ref)
		},
		"URLForService": func(org, proj, cmd string) string {
			parts := []string{"code.justin.tv", org, proj}
			if cmd != "" {
				parts = append(parts, cmd)
			}
			return routes.URIForProgramRef(&report_v1.ProgramRef{Name: path.Join(parts...)})
		},

		"add":   func(a, b int) int { return a + b },
		"scale": func(x, y float64) float64 { return x * y },
		"seq":   seq,
		"join":  strings.Join,

		"TimePercentile": timePercentile,
		"CompactDescr":   subtreeCompactDescription,
	})

	// set base templates
	var templates = map[string]string{
		"header":      "html/header.html",
		"base":        "html/base.html",
		"breadcrumbs": "html/components/breadcrumbs.html",
	}

	var err error
	for name, path := range templates {
		t, err = t.New(name).Parse(string(render.MustAsset(path)))
		if err != nil {
			return nil, err
		}
	}

	return t, nil
}

func renderTemplate(routes URIGenerator, assetPath string, data interface{}) ([]byte, error) {
	tmplSrc := string(render.MustAsset(assetPath))

	// TODO: don't re-parse the templates every time, if possible

	base, err := baseTemplate(routes)
	if err != nil {
		return nil, err
	}

	tmpl, err := base.New(assetPath).Parse(tmplSrc)
	if err != nil {
		return nil, err
	}

	var buf bytes.Buffer
	err = tmpl.Execute(&buf, data)
	if err != nil {
		return nil, err
	}

	return buf.Bytes(), nil
}

func RenderProgramReport(routes URIGenerator, home string, rep *report_v1.ProgramReport) ([]byte, error) {
	breadcrumbs := [][2]string{
		[2]string{home, "home"},
		[2]string{"", rep.GetProgram().GetName()},
	}

	data := struct {
		Report      *report_v1.ProgramReport
		Breadcrumbs [][2]string
		VegaSpec    interface{}
	}{
		Report:      rep,
		Breadcrumbs: breadcrumbs,
		VegaSpec:    serviceVegaSpec(rep.GetRootServerTimingDistribution()),
	}

	return renderTemplate(routes, "html/binary.html", data)
}

func RenderTransactionBasePage(routes URIGenerator) ([]byte, error) {
	return renderTemplate(routes, "html/tx.html", "")
}

func RenderProgramIndex(routes URIGenerator, programs []*report_v1.ProgramRef) ([]byte, error) {
	type project struct {
		Name       string
		IsACommand bool
		Commands   []string
	}
	type org struct {
		Name     string
		Projects []*project
	}
	type index struct {
		Orgs []*org
	}

	var data index

	orgs := make(map[string]map[string]*project)
	for _, ref := range programs {
		parts := strings.SplitN(ref.GetName(), "/", 4)
		if len(parts) < 3 {
			continue
		}
		if parts[0] != "code.justin.tv" {
			continue
		}

		orgName, projName, cmdName := parts[1], parts[2], strings.Join(parts[3:], "/")

		orgProjects, ok := orgs[orgName]
		if !ok {
			orgProjects = make(map[string]*project)
			orgs[orgName] = orgProjects
		}

		proj, ok := orgProjects[projName]
		if !ok {
			proj = &project{Name: projName}
			orgProjects[projName] = proj
		}

		proj.Commands = append(proj.Commands, cmdName)
		if cmdName == "" {
			proj.IsACommand = true
		}
	}

	for orgName, orgProjects := range orgs {
		o := &org{Name: orgName}
		data.Orgs = append(data.Orgs, o)
		for _, proj := range orgProjects {
			o.Projects = append(o.Projects, proj)
			sort.Strings(proj.Commands)
		}
		sort.Slice(o.Projects, func(i, j int) bool {
			return o.Projects[i].Name < o.Projects[j].Name
		})
	}

	sort.Slice(data.Orgs, func(i, j int) bool {
		return data.Orgs[i].Name < data.Orgs[j].Name
	})

	return renderTemplate(routes, "html/index.html", data)
}
