package loadroutes

import (
	"fmt"
	"reflect"
	"strings"
	"testing"

	"code.justin.tv/release/trace/rpc/alvin/internal/httpproto/google_api"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/protoc-gen-go/descriptor"
)

func TestGetUnregisteredExtension(t *testing.T) {
	t.Run("not extendable", func(t *testing.T) {
		_, err := getUnregisteredExtension(&google_api.HttpRule{}, e_Http)
		if have, want := fmt.Sprint(err), "not an extendable"; !strings.Contains(have, want) {
			t.Errorf("getUnregisteredExtension(&google_api.HttpRule{}, e_Http); err = %v", err)
		}
	})

	t.Run("GetExtension failure", func(t *testing.T) {
		pb := &descriptor.MethodOptions{}

		b := []byte("\x00")
		proto.SetRawExtension(pb, e_Http.Field, b)

		_, err := getUnregisteredExtension(pb, e_Http)
		if have, want := fmt.Sprint(err), "unexpected EOF"; !strings.Contains(have, want) {
			t.Errorf("getUnregisteredExtension(pb,  &proto.ExtensionDesc{...}); err = %v", err)
		}
	})

	withBadExtension := func(opts *descriptor.MethodOptions) *descriptor.MethodOptions {
		b := []byte("\x00")
		proto.SetRawExtension(opts, e_Http.Field, b)
		return opts
	}

	t.Run("", func(t *testing.T) {
		sd := &descriptor.ServiceDescriptorProto{
			Name: proto.String("Users"),
			Method: []*descriptor.MethodDescriptorProto{
				&descriptor.MethodDescriptorProto{
					Name:       proto.String("Find"),
					InputType:  proto.String("test.package.FindRequest"),
					OutputType: proto.String("test.package.FindResponse"),
					Options:    withBadExtension(&descriptor.MethodOptions{}),
				},
			},
		}
		specs, err := FromServiceDescriptor("", sd)
		if len(specs) > 0 {
			t.Errorf("FromServiceDescriptor(\"\", sd); specs = %#v", specs)
		}
		if err != nil {
			t.Errorf("FromServiceDescriptor(\"\", sd); err = %v", err)
		}
	})
}

type badProto struct{}

func (_ *badProto) Reset()                   {}
func (_ *badProto) String() string           { return "badProto" }
func (_ *badProto) ProtoMessage()            {}
func (_ *badProto) Marshal() ([]byte, error) { return nil, fmt.Errorf("badProto cannot marshal") }
func (_ *badProto) Unmarshal(_ []byte) error { return fmt.Errorf("badProto cannot unmarshal") }

func TestConvertProtoType(t *testing.T) {
	msg := &google_api.HttpRule{Pattern: &google_api.HttpRule_Get{Get: "/find"}}
	bad := (*badProto)(nil)

	t.Run("success", func(t *testing.T) {
		out, err := convertProtoType(msg, reflect.TypeOf(e_Http.ExtensionType))
		if err != nil {
			t.Errorf("convertProtoType; err = %v", err)
		}
		if have, want := out, msg; !reflect.DeepEqual(have, want) {
			t.Errorf("convertProtoType; %q != %q", have, want)
		}
	})

	t.Run("non-proto input", func(t *testing.T) {
		_, err := convertProtoType(nil, nil)
		if have, want := fmt.Sprint(err), "non-protobuf input type"; !strings.Contains(have, want) {
			t.Errorf("convertProtoType; err = %s", err)
		}
	})

	t.Run("bad proto input", func(t *testing.T) {
		_, err := convertProtoType(bad, nil)
		if have, want := fmt.Sprint(err), "badProto cannot marshal"; !strings.Contains(have, want) {
			t.Errorf("convertProtoType; err = %s", err)
		}
	})

	t.Run("non-proto output", func(t *testing.T) {
		_, err := convertProtoType(msg, reflect.TypeOf(0))
		if have, want := fmt.Sprint(err), "non-protobuf output type"; !strings.Contains(have, want) {
			t.Errorf("convertProtoType; err = %s", err)
		}
	})

	t.Run("bad proto output", func(t *testing.T) {
		_, err := convertProtoType(msg, reflect.TypeOf(bad))
		if have, want := fmt.Sprint(err), "badProto cannot unmarshal"; !strings.Contains(have, want) {
			t.Errorf("convertProtoType; err = %s", err)
		}
	})
}
