// +build integration

package test

import (
	"bytes"
	"encoding/json"
	"fmt"
	"log"
	"os/exec"
	"strings"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"code.justin.tv/systems/sandstorm/resource"
	"code.justin.tv/systems/sandstorm/testutil"
	. "github.com/smartystreets/goconvey/convey"
)

const (
	sandstormBin      = "sandstorm"
	sandstormAdminBin = "sandstorm-admin"

	subCommandPut          = "put"
	subCommandGet          = "get"
	subCommandGetVersions  = "get-versions"
	subCommandGetEncrypted = "get-encrypted"
	subCommandCreate       = "create"
	subCommandRevert       = "revert"
	subCommandUpdate       = "update"

	optionDNB          = "--dnb"
	optionEnvironment  = "--environment"
	optionARN          = "-r"
	optionClass        = "-c"
	optionNoNewLine    = "-n"
	optionDetailed     = "-d"
	optionVersion      = "--version"
	optionAutogenerate = "--autogenerate"
)

var config *resource.Config

func createTestManager() *manager.Manager {
	res, err := resource.GetConfigForEnvironment("testing")
	if err != nil {
		log.Fatal(err)
	}

	creds := resource.AWSCredentials(res.AwsRegion, nil, res.RoleArn)

	return manager.New(manager.Config{
		AWSConfig:   resource.AWSConfig(creds),
		ActionUser:  resource.GetAWSIdentity(creds),
		Environment: "testing",
	})
}

func readConfig() *resource.Config {
	if config != nil {
		return config
	}
	config, err := resource.GetConfigForEnvironment("testing")
	if err != nil {
		log.Fatal(err.Error())
	}
	return config
}

func runCliCommand(command string, args []string) (bytes.Buffer, error) {
	_, err := exec.LookPath(command)
	if err != nil {
		log.Fatal(fmt.Printf("Cannot find %s \n", command))
	}

	cmd := exec.Command(command, args...)
	var out bytes.Buffer
	var stderr bytes.Buffer
	cmd.Stdout = &out
	cmd.Stderr = &stderr

	err = cmd.Run()

	if err != nil {
		return out, err
	}
	return out, nil
}

func runSandstormCliCommand(args []string) (bytes.Buffer, error) {
	return runCliCommand(sandstormBin, args)
}

func getTestOptionsArgs(testConfig *resource.Config) []string {
	return []string{
		optionEnvironment, "testing",
		optionARN, testConfig.RoleArn,
	}
}

func deleteSecretUsingAdmin(secretName string) {
	command := sandstormAdminBin
	subCommand := "delete-secret"
	testConfig := readConfig()
	args := []string{subCommand, testConfig.TableName, secretName}

	_, err := runCliCommand(command, args)
	if err != nil {
		log.Printf("failed to delete secret %s, Error: %s", secretName, err.Error())
	}
}

func getSecretEncrypted(secretName string) (*manager.Secret, error) {
	testConfig := readConfig()
	args := []string{subCommandGetEncrypted, optionEnvironment, "testing", optionARN, testConfig.RoleArn, secretName}
	contents, err := runSandstormCliCommand(args)
	if err != nil {
		return nil, err
	}
	secret := new(manager.Secret)
	err = json.Unmarshal(contents.Bytes(), &secret)
	if err != nil {
		return nil, err
	}

	return secret, nil
}

func getSecret(secretName string) string {
	testConfig := readConfig()
	args := []string{subCommandGet, optionEnvironment, "testing", optionARN, testConfig.RoleArn, optionNoNewLine, secretName}
	contents, err := runSandstormCliCommand(args)
	if err != nil {
		log.Fatalf("Error: %s", err.Error())
		return ""
	}

	return contents.String()
}

func getSecretVersion(secretName string, version string) (string, error) {
	testConfig := readConfig()
	args := []string{
		subCommandGet,
		optionEnvironment, "testing",
		optionARN, testConfig.RoleArn,
		optionNoNewLine,
		optionVersion, version,
		secretName,
	}

	contents, err := runSandstormCliCommand(args)
	if err != nil {
		return "", err
	}
	return contents.String(), nil
}

func TestSandstormCLIGetSecret(t *testing.T) {
	Convey("sandstorm-cli", t, func() {
		m := createTestManager()
		testConfig := readConfig()
		secretName := testutil.GetRandomSecretName(t)
		secretValue := "SomeSecret"

		Convey("create", func() {
			// create a secret for furthur testing
			baseArgs := []string{subCommandCreate,
				optionEnvironment, "testing",
				optionARN, testConfig.RoleArn,
			}

			Convey("Create a new secret with no class and do not broadcast flag.", func() {
				out, err := runSandstormCliCommand(append(baseArgs, []string{secretName, secretValue}...))

				So(err, ShouldBeNil)
				So(out, ShouldNotBeNil)

				secret, err := getSecretEncrypted(secretName)
				So(err, ShouldBeNil)
				So(secret, ShouldNotBeNil)
				actualSecret := getSecret(secretName)
				So(actualSecret, ShouldEqual, secretValue)
				So(secret.Class, ShouldEqual, manager.ClassDefault)
				So(secret.DoNotBroadcast, ShouldBeFalse)

			})

			Convey("Create a new secret with class specified.", func() {
				args := append(baseArgs, []string{optionClass, "3", secretName, secretValue}...)

				out, err := runSandstormCliCommand(args)

				So(err, ShouldBeNil)
				So(out, ShouldNotBeNil)

				secret, err := getSecretEncrypted(secretName)
				So(err, ShouldBeNil)
				So(secret, ShouldNotBeNil)
				actualSecret := getSecret(secretName)
				So(actualSecret, ShouldEqual, secretValue)

				So(secret.Class, ShouldEqual, manager.ClassRestricted)
				So(secret.DoNotBroadcast, ShouldBeFalse)

			})

			Convey("Create a new secret with --autogenerate.", func() {
				args := append(baseArgs, []string{
					optionAutogenerate,
					"20",
					secretName,
				}...)
				out, err := runSandstormCliCommand(args)
				t.Logf(out.String())
				So(err, ShouldBeNil)
				So(out, ShouldNotBeNil)

				actualSecret, err := m.Get(secretName)
				So(err, ShouldBeNil)
				So(len(actualSecret.Plaintext), ShouldEqual, 28)
			})

			Convey("Create a new secret with --autogenerate and value should error out.", func() {
				args := append(baseArgs, []string{
					optionAutogenerate,
					"20",
					secretName,
					"my-secret-value",
				}...)
				out, err := runSandstormCliCommand(args)
				So(err, ShouldNotBeNil)
				So(out, ShouldNotBeNil)
			})

			Convey("Create a new secret with do_not_broadcast flag set to true.", func() {
				args := append(baseArgs, []string{optionDNB, secretName, secretValue}...)
				out, err := runSandstormCliCommand(args)

				So(err, ShouldBeNil)
				So(out, ShouldNotBeNil)

				secret, err := getSecretEncrypted(secretName)
				So(err, ShouldBeNil)
				So(secret, ShouldNotBeNil)
				actualSecret := getSecret(secretName)
				So(actualSecret, ShouldEqual, secretValue)

				So(secret.Class, ShouldEqual, manager.ClassDefault)
				So(secret.DoNotBroadcast, ShouldBeTrue)

				Convey("Get created secret by version.", func() {
					versions, err := m.GetVersionsEncrypted(secretName, 10, 0)
					So(err, ShouldBeNil)
					So(len(versions.Secrets), ShouldEqual, 1)
					version := versions.Secrets[0]

					out, err := getSecretVersion(secretName, fmt.Sprintf("%d", version.UpdatedAt))
					So(err, ShouldBeNil)
					So(out, ShouldEqual, fmt.Sprintf("%s", secretValue))
				})
			})

			Convey("Getting a not-existing secret version should error outGetting a not-existing secret version should error out.", func() {
				out, err := getSecretVersion(secretName, "666")
				So(err, ShouldNotBeNil)
				So(out, ShouldEqual, "")
			})
		})

		Convey("put", func() {
			Convey("Should write notice to STDERR if STDIN is TTY", func() {
				var stderr bytes.Buffer
				cmd := exec.Command(
					sandstormBin,
					"put",
					optionEnvironment, "testing",
					optionARN, testConfig.RoleArn,
					secretName,
				)
				cmd.Stderr = &stderr
				err := cmd.Start()
				So(err, ShouldBeNil)

				done := make(chan error, 1)
				go func() {
					done <- cmd.Wait()
				}()
				// wait a max of 100ms before killing the process
				select {
				case <-time.After(100 * time.Millisecond):
					if err := cmd.Process.Kill(); err != nil {
						log.Fatal("failed to kill: ", err)
					}
				case <-done:
					t.Logf("Exited before kill.")
				}

				So(strings.HasPrefix(string(stderr.Bytes()), "Listening on STDIN for secret value.\n"), ShouldBeTrue)
			})
		})

		Convey("get-versions", func() {
			Convey("Getting versions of non-existing secret should return no versions.", func() {
				versions, err := runSandstormCliCommand([]string{
					subCommandGetVersions,
					optionEnvironment, "testing",
					optionARN, testConfig.RoleArn,
					secretName,
				})
				So(err, ShouldNotBeNil)
				So(string(versions.Bytes()), ShouldEqual, "")
			})

			Convey("Getting versions of existing secret.", func() {
				So(m.Post(&manager.Secret{
					Name:      secretName,
					Plaintext: []byte(secretValue),
				}), ShouldBeNil)

				expectedVersions, err := m.GetVersionsEncrypted(secretName, 10, 0)
				So(err, ShouldBeNil)
				So(len(expectedVersions.Secrets), ShouldEqual, 1)
				expectedVersion := expectedVersions.Secrets[0]

				versions, err := runSandstormCliCommand([]string{
					subCommandGetVersions,
					optionEnvironment, "testing",
					optionARN, testConfig.RoleArn,
					secretName,
				})
				So(err, ShouldBeNil)
				So(string(versions.Bytes()), ShouldEqual, fmt.Sprintf("%d\n", expectedVersion.UpdatedAt))
			})
		})

		Convey("revert", func() {
			Convey("Non integer values should error out.", func() {
				args := []string{subCommandRevert, secretName, "nonIntegerValue"}
				args = append(args, getTestOptionsArgs(testConfig)...)
				_, err := runSandstormCliCommand(args)
				So(err, ShouldNotBeNil)
			})

			Convey("Non-existing version should error out.", func() {
				So(m.Post(&manager.Secret{
					Name:      secretName,
					Plaintext: []byte(secretValue),
				}), ShouldBeNil)
				versions, err := m.GetVersionsEncrypted(secretName, 10, 0)
				So(err, ShouldBeNil)
				So(len(versions.Secrets), ShouldEqual, 1)
				originalVersion := versions.Secrets[0]

				args := []string{subCommandRevert, secretName, fmt.Sprintf("%d", originalVersion.UpdatedAt+1)}
				args = append(args, getTestOptionsArgs(testConfig)...)
				_, err = runSandstormCliCommand(args)
				So(err, ShouldNotBeNil)
			})

			Convey("Existing version", func() {
				So(m.Post(&manager.Secret{
					Name:      secretName,
					Plaintext: []byte(secretValue),
				}), ShouldBeNil)

				versions, err := m.GetVersionsEncrypted(secretName, 10, 0)
				So(err, ShouldBeNil)
				So(len(versions.Secrets), ShouldEqual, 1)
				originalSecret := versions.Secrets[0]

				args := []string{subCommandRevert, secretName, fmt.Sprintf("%d", originalSecret.UpdatedAt+1)}
				args = append(args, getTestOptionsArgs(testConfig)...)
				_, err = runSandstormCliCommand(args)
				So(err, ShouldNotBeNil)

				secret, err := m.Get(secretName)
				So(err, ShouldBeNil)
				So(secret.UpdatedAt, ShouldEqual, originalSecret.UpdatedAt)
			})
		})

		Convey("update", func() {
			// create a secret for furthur testing
			createArgs := []string{subCommandCreate,
				optionEnvironment, "testing",
				optionARN, testConfig.RoleArn,
				secretName,
				secretValue,
			}

			out, err := runSandstormCliCommand(createArgs)

			So(err, ShouldBeNil)
			So(out, ShouldNotBeNil)
			Convey("Should store secrets correctly with default values.", func() {
				secret, err := getSecretEncrypted(secretName)
				So(err, ShouldBeNil)
				So(secret, ShouldNotBeNil)

				actualSecret := getSecret(secretName)
				So(actualSecret, ShouldEqual, secretValue)
				So(secret.Class, ShouldEqual, manager.ClassDefault)
				So(secret.DoNotBroadcast, ShouldBeFalse)

				updateArgs := []string{subCommandUpdate,
					optionEnvironment, "testing",
					optionARN, testConfig.RoleArn,
				}

				Convey("should set 'class' to Restricted", func() {
					args := append(updateArgs, []string{optionClass, "3", secretName, secretValue}...)
					_, err = runSandstormCliCommand(args)

					secret, err := getSecretEncrypted(secretName)
					So(err, ShouldBeNil)
					So(secret, ShouldNotBeNil)

					So(secret.Class, ShouldEqual, manager.ClassRestricted)
				})

				Convey("should set do_not_broadcast to true.", func() {
					args := append(updateArgs, []string{optionDNB, secretName, secretValue}...)
					output, err := runSandstormCliCommand(args)
					So(err, ShouldBeNil)
					So(output, ShouldNotBeNil)
					secret, err := getSecretEncrypted(secretName)
					So(err, ShouldBeNil)
					So(secret, ShouldNotBeNil)
					So(secret.DoNotBroadcast, ShouldBeTrue)
				})
			})
		})

		Reset(func() {
			So(m.Delete(secretName), ShouldBeNil)
		})
	})
}
