resource "aws_sns_topic" "pagerduty" {
  name = "sandstorm-pagerduty-${var.env}"
}

resource "aws_sns_topic_subscription" "pagerduty" {
  topic_arn              = "${aws_sns_topic.pagerduty.arn}"
  protocol               = "https"
  endpoint               = "${var.pagerduty_webhook_url}"
  endpoint_auto_confirms = true
}

resource "aws_cloudwatch_metric_alarm" "changelog-lambda-metric" {
  alarm_name          = "changelog-lambda-errors-${var.env}"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "1"
  metric_name         = "Errors"
  namespace           = "AWS/Lambda"
  period              = "${var.lambda_alarm_period}"
  statistic           = "Sum"
  threshold           = "${var.lambda_alarm_threshold}"
  alarm_description   = "Failed to send events to ${var.changelog_endpoint}"

  alarm_actions = [
    "${aws_sns_topic.pagerduty.arn}",
  ]

  ok_actions = [
    "${aws_sns_topic.pagerduty.arn}",
  ]

  dimensions {
    FunctionName = "${aws_lambda_function.lambda.function_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "sandstorm-lambda-metric" {
  alarm_name          = "sandstorm-sns-publish-errors-${var.env}"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "1"
  metric_name         = "Errors"
  namespace           = "AWS/Lambda"
  period              = "${var.lambda_alarm_period}"
  statistic           = "Sum"
  threshold           = "${var.lambda_alarm_threshold}"
  alarm_description   = "Failed to send publish secret update to sns ${aws_sns_topic.sandstorm.arn}"

  alarm_actions = [
    "${aws_sns_topic.pagerduty.arn}",
  ]

  ok_actions = [
    "${aws_sns_topic.pagerduty.arn}",
  ]

  dimensions {
    FunctionName = "${aws_lambda_function.sandstorm-lambda.function_name}"
  }
}

resource "aws_cloudwatch_metric_alarm" "kms-high-usage" {
  alarm_name          = "sandstorm-kms-high-usage-${var.env}"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "1"
  metric_name         = "DecryptAttempted"
  namespace           = "SandstormKMS"
  period              = "60"
  statistic           = "SampleCount"
  threshold           = "${floor(local.kms_tpm_limit / 2)}"
  alarm_description   = "Sandstorm KMS usage is over 50% of the limit: ${local.runbook_url}"
  treat_missing_data  = "notBreaching"

  alarm_actions = [
    "${aws_sns_topic.pagerduty.arn}",
  ]

  ok_actions = [
    "${aws_sns_topic.pagerduty.arn}",
  ]

  dimensions {
    KMSKeyARN = "${aws_kms_key.sandstorm.arn}"
  }
}
