data "aws_iam_policy_document" "lambda-stream-permissions" {
  statement {
    actions = [
      "dynamodb:GetRecords",
      "dynamodb:GetShardIterator",
      "dynamodb:DescribeStream",
      "dynamodb:ListStreams",
    ]

    resources = [
      "${aws_dynamodb_table.sandstorm-audit.stream_arn}",
    ]

    effect = "Allow"
  }
}

resource "aws_iam_role" "lambda" {
  name               = "sandstorm-changelog-lambda-${var.env}"
  assume_role_policy = "${data.aws_iam_policy_document.lambda-assume-role.json}"
}

resource "aws_iam_policy" "lambda" {
  name        = "sandstorm-changelog-lambda-${var.env}"
  path        = "/"
  description = "lambda access to sandstorm table dynamodb stream"
  policy      = "${data.aws_iam_policy_document.lambda-stream-permissions.json}"
}

resource "aws_iam_policy" "lambda-logs-permission" {
  name        = "sandstorm-lambda-logs-${var.env}"
  path        = "/"
  description = "lambda access to cloudwatch logs"
  policy      = "${data.aws_iam_policy_document.lambda-logs-permission.json}"
}

resource "aws_iam_policy_attachment" "lambda-attachment" {
  name = "sandstorm-changelog-lambda-attachment-${var.env}"

  roles = [
    "${aws_iam_role.lambda.name}",
  ]

  policy_arn = "${aws_iam_policy.lambda.arn}"
}

resource "aws_iam_policy_attachment" "lambda-logs-attachment" {
  name = "sandstorm-lambda-logs-attachment-${var.env}"

  roles = [
    "${aws_iam_role.lambda.name}",
    "${aws_iam_role.sandstorm-lambda-permissions.name}",
  ]

  policy_arn = "${aws_iam_policy.lambda-logs-permission.arn}"
}

resource "aws_lambda_function" "lambda" {
  function_name = "changelog-${var.env}"
  role          = "${aws_iam_role.lambda.arn}"
  handler       = "main.handler"
  s3_bucket     = "${var.lambda_s3_bucket}"
  s3_key        = "changelog-lambda/${local.lambda_source_s3_key}.zip"
  runtime       = "python2.7"
  timeout       = "30"

  vpc_config {
    subnet_ids         = "${var.subnet_ids}"
    security_group_ids = "${var.security_group_ids}"
  }

  environment {
    variables = {
      SANDSTORM_ENVIRONMENT        = "${var.env}"
      SANDSTORM_CHANGELOG_ENDPOINT = "${var.changelog_endpoint}/api/events"
    }
  }

  depends_on = ["aws_iam_role.lambda"]
}

resource "aws_lambda_event_source_mapping" "event_source_mapping" {
  batch_size        = 1
  event_source_arn  = "${aws_dynamodb_table.sandstorm-audit.stream_arn}"
  enabled           = true
  function_name     = "${aws_lambda_function.lambda.arn}"
  starting_position = "LATEST"
}

resource "aws_iam_policy" "lambda-vpc" {
  name        = "sandstorm-lambda-vpc-${var.env}"
  path        = "/"
  description = "permission to create describe and delete network interfaces"
  policy      = "${data.aws_iam_policy_document.lambda-ec2-vpc.json}"
}

resource "aws_iam_policy_attachment" "vpc-lambda" {
  name = "changelog-lambda-vpc-${var.env}"

  roles = [
    "${aws_iam_role.lambda.name}",
  ]

  policy_arn = "${aws_iam_policy.lambda-vpc.arn}"
}
