resource "aws_lambda_function" "cleanup-sns-sqs-subs" {
  function_name = "cleanup_sns_sqs_subs-${var.env}"
  role          = "${aws_iam_role.cleanup-sns-sqs-subs.arn}"
  handler       = "main.cleanup_sns_sqs_subs"
  s3_bucket     = "${var.lambda_s3_bucket}"
  s3_key        = "cleanup-lambdas/${local.lambda_source_s3_key}.zip"
  runtime       = "python3.6"
  timeout       = "300"

  environment {
    variables {
      SANDSTORM_SNS_TOPIC_ARN = "${aws_sns_topic.sandstorm.arn}"
    }
  }
}

resource "aws_iam_role" "cleanup-sns-sqs-subs" {
  name               = "cleanup-sns-sqs-subs-lambda-${var.env}"
  assume_role_policy = "${data.aws_iam_policy_document.lambda-assume-role.json}"
}

resource "aws_iam_role_policy" "cleanup-sns-sqs-subs" {
  name   = "cleanup-sns-sqs-subs"
  role   = "${aws_iam_role.cleanup-sns-sqs-subs.id}"
  policy = "${data.aws_iam_policy_document.cleanup-sns-sqs-subs-rp.json}"
}

data "aws_iam_policy_document" "cleanup-sns-sqs-subs-rp" {
  statement {
    sid = "SNS"

    actions = [
      "sns:ListSubscriptionsByTopic",
      "sns:Unsubscribe",
    ]

    resources = [
      "${aws_sns_topic.sandstorm.arn}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "SQS"

    actions = [
      "sqs:GetQueueUrl",
      "sqs:DeleteQueue",
    ]

    resources = [
      "arn:aws:sqs:*:${var.account_id}:sandstorm-*",
    ]

    effect = "Allow"
  }

  statement {
    sid = "Cloudwatch"

    actions = [
      "cloudwatch:GetMetricStatistics",
      "cloudwatch:PutMetricData",
    ]

    resources = [
      "*",
    ]

    effect = "Allow"
  }

  statement {
    sid = "LambdaInvoke"

    actions = [
      "lambda:InvokeFunction",
    ]

    resources = [
      "${aws_lambda_function.cleanup-sns-sqs-subs.arn}",
    ]

    effect = "Allow"
  }

  statement {
    sid = "Logging"

    actions = [
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]

    resources = [
      "arn:aws:logs:*:*:*",
    ]

    effect = "Allow"
  }
}

module "cleanup-sns-sqs-subs-lambda-cron-trigger" {
  source         = "./lambda-cron-trigger"
  scheduler_name = "${aws_lambda_function.cleanup-sns-sqs-subs.function_name}-weekly"
  function_arn   = "${aws_lambda_function.cleanup-sns-sqs-subs.arn}"
  sns_alarm_arn  = "${aws_sns_topic.pagerduty.arn}"
  schedule       = "rate(7 days)"
}
